//////////////////////////////////////////////////////////////////////
//   CANdriver.h
//////////////////////////////////////////////////////////////////////
//
//   Description: 
//	 CAN driver for dsPIC30F devices		
//
//   Author: Dominic Letourneau, Pierre Lepage
//   Creation Date: 2006/11/06
//
//   Modification history:
//
//  Name					Date        Descrition
//	Dominic Letourneau		2006/11/06	Modification for dsPICs					
//  Dominic Letourneau		2007/01/23	Added new boot configuration and
//										eeprom read & write
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius / RoboMotio  2007
//////////////////////////////////////////////////////////////////////

#ifndef _CAN_DRIVER_H_
#define _CAN_DRIVER_H_

#include "CANShared.h"

/* LABORIUS VARIABLE SECTION ################################################################# */
/*
MESSAGE PRIORITY DEFINITIONS
*/
#define CAN_PRIORITY_HIGHEST     0
#define CAN_PRIORITY_HIGH        1
#define CAN_PRIORITY_MEDIUM_HIGH 2
#define CAN_PRIORITY_MEDIUM      3
#define CAN_PRIORITY_MEDIUM_LOW  4
#define CAN_PRIORITY_LOW         5
#define CAN_PRIORITY_LOWEST      6
#define CAN_PRIORITY_EVENTS      7

/*
MESSAGE TYPE DEFINITIONS
*/
#define CAN_TYPE_NOTHING				  0x00
#define CAN_TYPE_EMERGENCY                0x01
#define CAN_TYPE_ACTUATOR_HIGH_PRIORITY   0x02
#define CAN_TYPE_SENSOR_HIGH_PRIORITY     0x04
#define CAN_TYPE_ACTUATOR_LOW_PRIORITY    0x08
#define CAN_TYPE_SENSOR_LOW_PRIORITY      0x10
#define CAN_TYPE_REQUEST_DATA             0x20
#define CAN_TYPE_USER2                    0x40
#define CAN_TYPE_EVENTS                   0x80
#define CAN_TYPE_EVERYTHING               0xFF




/*
EMERGENCY MESSAGE COMMANDS (TYPE_EMERGENCY)
*/
#define CAN_EMERGENCY_CMD_RESET        0x00
#define CAN_EMERGENCY_CMD_STOP         0x01
#define CAN_EMERGENCY_CMD_DISCONNECT   0x02
#define CAN_EMERGENCY_CMD_RECONNECT    0x03
#define CAN_EMERGENCY_CMD_PROGRAM      0x04



/*
BOOT COMMANDS
*/
#define CAN_REQUEST_READ                0x01
#define CAN_REQUEST_WRITE               0x00
#define CAN_REQUEST_EEPROM              0x01
#define CAN_REQUEST_RAM                 0x00

/*
EVENTS MESSAGE COMMANDS (TYPE EVENTS)
*/

#define CAN_EVENTS_CMD_ALIVE           	0x00
#define CAN_EVENTS_CMD_REGISTER        	0x01  // unused
#define CAN_EVENTS_CMD_SILENCE			0x02

/*
EVENTS MESSAGE COMMANDS (TYPE SENSOR_HIGH_PRIORITY)
*/

/*
CAN ADDRESS DEFINITIONS
*/
#define CAN_ADDRESS_BROADCAST 0xFF

/*
STRUCTURES
*/

typedef struct _message{
   unsigned char msg_priority;
   unsigned char msg_type;
   unsigned char msg_cmd;
   unsigned char msg_dest;
   unsigned char msg_read_write;
   unsigned char msg_eeprom_ram;
   unsigned char msg_data_length;
   unsigned char msg_data[8];
   unsigned char msg_remote;
   unsigned char msg_filter_hit;
   unsigned char msg_overflow;
} LABORIUS_MESSAGE;

typedef struct _filter {
   unsigned char filter_priority;
   unsigned char filter_type;
   unsigned char filter_cmd;
   unsigned char filter_dest;
} LABORIUS_FILTER;

typedef struct _mask {
   unsigned char mask_priority;
   unsigned char mask_type;
   unsigned char mask_cmd;
   unsigned char mask_dest;
} LABORIUS_MASK;

//THIS STRUCTURE HOLDS BOOT INFORMATION
//IN THE FIRST 6 BYTES OF EEPROM, STORED IN RAM
/////////////////////////////////////////////////////////////////
//    MEMORY ORGANISATION
//
//    VARIABLE                         OFFSET      MEANING
//    -----------------------------------------------------------
//    DATA_FLOW_TABLE                  0           MODULE STATE
//    DATA_FLOW_TABLE                  1           PROJECT ID
//    DATA_FLOW_TABLE                  2           MODULE ID (CAN ADDRESS)
//    DATA_FLOW_TABLE                  3           CODE VERSION 
//    DATA_FLOW_TABLE                  4           TABLE VERSION
//    DATA_FLOW_TABLE                  5           BOOT_DELAY (500ms increments)
//    DATA_FLOW_TABLE                  6           unused
//    DATA_FLOW_TABLE                  7           unused
/////////////////////////////////////////////////////////////////
typedef struct _BootConfig
{
	//WILL BE WRITTEN TO EEPROM
	unsigned char module_state;
	unsigned char project_id;
	unsigned char module_id;
	unsigned char code_version;
	unsigned char table_version;
	unsigned char boot_delay;

	//READ ONLY DEVICE ID
	unsigned char devid_low;
	unsigned char devid_high;		
} BootConfig;

//MODULE STATES
enum {BOOT_IDLE=0x0A, BOOT_PROGRAM_FLASH=0x01, BOOT_PROGRAM_EEPROM=0x02, BOOT_NORMAL=0x05};


// DATA_FLOW_TABLE Stuff
// ADDR, OFFSET, LIMITS on memory
#define DATA_FLOW_TABLE_SIZE		(sizeof(GlobalVariables))
/*

//TABLE VERSION 1 (deprecated)

#define DATA_FLOW_ENTRY             0x08 //8 bytes
#define MAX_RAM_DATA_ENTRY          0x80 //128 bytes
#define MAX_EEPROM_DATA_ENTRY       0x80 //128 bytes
#define BIT_SET_LIMITS              0x07 //maximum index
#define SECTION_MASK                0x0F 
#define SUB_SECTION_MASK            0xF0
#define SUB_SECTION_SHIFT           0x04 
#define NUM_SECTION					16   //16 pages of 8 bytes

//READ_WRITE MEMORY
#define READ_WRITE_1_BYTE           0x08
#define READ_WRITE_2_BYTE           0x09
#define READ_WRITE_3_BYTE           0x0A
#define READ_WRITE_4_BYTE           0x0B
#define READ_WRITE_5_BYTE           0x0C
#define READ_WRITE_6_BYTE           0x0D
#define READ_WRITE_7_BYTE           0x0E
#define READ_WRITE_8_BYTE           0x0F

#define READ_WRITE_BIT_0            0x00
#define READ_WRITE_BIT_1            0x01
#define READ_WRITE_BIT_2            0x02
#define READ_WRITE_BIT_3            0x03
#define READ_WRITE_BIT_4            0x04
#define READ_WRITE_BIT_5            0x05
#define READ_WRITE_BIT_6            0x06
#define READ_WRITE_BIT_7            0x07
*/

//value to put in mask field to accept all incoming id's
#define CAN_MASK_ACCEPT_ALL   0

/* LABORIUS FUNCTIONS' PROTOTYPE ########################################################################*/
void can_init(LABORIUS_FILTER *filter,LABORIUS_MASK *mask);
void can_apply_accept_mask(LABORIUS_MASK *mask, unsigned char mask_id);
void can_apply_filter(LABORIUS_FILTER *filter, unsigned char filter_id);
char can_send_laborius_packet(LABORIUS_MESSAGE *message);
char can_recv_laborius_packet(LABORIUS_MESSAGE *message);
void can_send_im_alive(unsigned char addr, BootConfig *config);
void can_transceiver(unsigned char can_addr);

// Every program should implement this function, which is a callback function from the can_transceiver general
// CAN message handling
extern void proc_message(LABORIUS_MESSAGE *message);


//Memory utilities
void can_write_eeprom(unsigned int index, unsigned int data);
unsigned int can_read_eeprom(unsigned int index);
unsigned char read_data_flow_table_v2(unsigned int offset, unsigned char mem_type, unsigned char *buffer, unsigned int size);
unsigned char write_data_flow_table_v2(unsigned int offset, unsigned char mem_type, unsigned char *buffer, unsigned int size);
unsigned char eeprom_hit(void);

BootConfig* can_get_boot_config();
void can_read_boot_config(BootConfig *config);
void can_write_boot_config(BootConfig *config);

//Reserved memory for CAN tables
extern unsigned char* const DATA_FLOW_TABLE;


#endif
