#include "Device.h"
#include "CANShared.h"


/* The DSP library is suboptimal. It take 4 cycle to do a MAC while 
 * we can do it in only 1 cycle ... 
 */ 
#include "dotprod.h"


#define RST_N_COEFF 14
#define ADMIT_N_COEFF 5

/* First the R coeff already negated, then the S coeff also negated
 * then the T coeff. */

static int __attribute__((section(".xbss, bss, xmemory"))) rst_coeff[RST_N_COEFF+1];
/* Why +1 ? the MAC operation prefetch the data, so we must have valid access one
 * int after the last used one. */


/* u(k-1), u(k-2), u(k-3), u(k-4), y(k), y(k-1), y(k-2), y(k-3),
 * y(k-4), yc(k), yc(k-1), yc(k-2), yc(k-3), yc(k-4) */
static int __attribute__((section(".ybss, bss, ymemory"))) rst_state[RST_N_COEFF+1];


/* Frist the denominator, already negated, then the numerator */
//static int __attribute__((section(".xbss, bss, xmemory"))) admit_coeff[ADMIT_N_COEFF+1];

/* u(k-1), u(k-2), y(k), y(k-1) y(k-2)*/
//static int __attribute__((section(".ybss, bss, ymemory"))) admit_state[ADMIT_N_COEFF+1];

static float admit_coeff[ADMIT_N_COEFF] = {1.0, 0.0,			/* 1/(0.3s) */
										   0.0, -0.04257479, 0.};
static float admit_state[ADMIT_N_COEFF];
static float admit_state_arw[ADMIT_N_COEFF];

static float rst_coeff_f[RST_N_COEFF] = {1.7397, -1.3993, 0.5602, -0.1176,               /* -R */
										 20.4667, -23.2917, -10.0833, 22.5250, -11.1500, /* -S */ 
										 17.2083, -30.9833, 20.9083, -6.2750, 0.7083};   /*  T */
static float rst_state_f[RST_N_COEFF];

/* Read the RST coeff in the shared memory then update our internal one */
void rst_update_coeff(SharedVariables* variables) {
	__asm__ volatile ("disi #0x3FFF");
 	rst_coeff[0] = variables->r1;
	rst_coeff[1] = variables->r2;
	rst_coeff[2] = variables->r3;
	rst_coeff[3] = variables->r4;

	rst_coeff[4] = variables->s0;
	rst_coeff[5] = variables->s1;
	rst_coeff[6] = variables->s2;
	rst_coeff[7] = variables->s3;
	rst_coeff[8] = variables->s4;

	rst_coeff[9] = variables->t0;
	rst_coeff[10] = variables->t1;
	rst_coeff[11] = variables->t2;
	rst_coeff[12] = variables->t3;
	rst_coeff[13] = variables->t4;
	__asm__ volatile ("disi #0x0000");
}

void rst_reset_state(void) {
	int i = 0;

	__asm__ volatile ("disi #0x3FFF");
	for(; i < RST_N_COEFF; i++) 
		rst_state[i] = 0;
	__asm__ volatile ("disi #0x0000");
}

int rst_do_step_float(SharedVariables* variables) {
	float output = 0;
	int i;

	rst_state_f[4] = variables->MesPoint;  /* Position */
	rst_state_f[9] = variables->RefPoint;  /* Commande */
	for(i = 0; i < RST_N_COEFF; i++) 
		output += rst_coeff_f[i] * rst_state_f[i];

	rst_state_f[8] = rst_state_f[7];
	rst_state_f[7] = rst_state_f[6];
	rst_state_f[6] = rst_state_f[5];
	rst_state_f[5] = rst_state_f[4];	

	rst_state_f[13] = rst_state_f[12];
	rst_state_f[12] = rst_state_f[11];
	rst_state_f[11] = rst_state_f[10];
	rst_state_f[10] = rst_state_f[9];	

	rst_state_f[3] = rst_state_f[2];
	rst_state_f[2] = rst_state_f[1];
	rst_state_f[1] = rst_state_f[0];
	rst_state_f[0] = output;
	return (int) output;
}
	

int rst_do_step(SharedVariables* variables) {
	int corcon_save;
	int rcount_save;
	int output;
	int old_0;
	/* First, add the current position and command to the state */
	rst_state[4] = variables->MesPoint; /* Position */
	rst_state[9] = variables->RefPoint;  /* Commande */
	
	__asm__ volatile ("disi #0x3FFF");
	corcon_save = CORCON;
	rcount_save = RCOUNT;
	/* clear the accumulator */
	__asm__ volatile ("clr A");
	/* Set the DSP in 1.15 fractional format */
	CORCONbits.IF = 1;
	CORCONbits.ACCSAT = 1; /* 9.31 saturation */
	CORCONbits.US = 0;
	CORCONbits.SATA = 1; /* Enable saturation */
	CORCONbits.SATDW = 1; 	/* Enable saturation on data space write */
	/* Do the RST calc. */
	dot_product(RST_N_COEFF-1, rst_coeff, rst_state);

	/* Read the result */

	/* First for the RST state */
	old_0 = rst_state[0];
	rst_state[0] = read_acca(variables->rst_shift);

	/* And now for the PWM output */
	output = read_acca(0);

	RCOUNT = rcount_save;
	CORCON = corcon_save;
	__asm__ volatile ("disi #0x0000");

/* 	Update the variables */
	rst_state[8] = rst_state[7];
	rst_state[7] = rst_state[6];
	rst_state[6] = rst_state[5];
	rst_state[5] = rst_state[4];	

	rst_state[13] = rst_state[12];
	rst_state[12] = rst_state[11];
	rst_state[11] = rst_state[10];
	rst_state[10] = rst_state[9];	

	rst_state[3] = rst_state[2];
	rst_state[2] = rst_state[1];
	rst_state[1] = old_0;

	return output;
}

void rst_init_pos(int pos) {
	float p = pos;

	/* Init our rst, so we can enable it while the device is not
	 * at 0 */
	rst_state_f[0] = 0;
	rst_state_f[1] = 0;
	rst_state_f[2] = 0;
	rst_state_f[3] = 0;
	
	rst_state_f[4] = p;
	rst_state_f[5] = p;
	rst_state_f[6] = p;
	rst_state_f[7] = p;
	rst_state_f[8] = p;
	rst_state_f[9] = p;
	rst_state_f[10] = p;
	rst_state_f[11] = p;
	rst_state_f[12] = p;
	rst_state_f[13] = p;
	
}


/*********************************************
 *           Admittance computation          *
 *********************************************/

//void do_admit_calc(SharedVariables* variables) {

	/* Z transorm shape of the admittance:
	 *    A + B z^-1 
	 * ----------------
	 * 1 + C z^-1 + D z^-2 
	 */
	
//	int corcon_save;
//	int rcount_save;
//	int output;

//	admit_state[2] = variables->EXT1Value; /* Torque */

//	__asm__ volatile ("disi #0x3FFF");
//	corcon_save = CORCON;
//	rcount_save = RCOUNT;
	/* clear the accumulator */
//	__asm__ volatile ("clr A");	
	/* Set the DSP in 1.15 fractional format */
//	CORCONbits.IF = 1;
//	CORCONbits.ACCSAT = 1; /* 9.31 saturation */
//	CORCONbits.US = 0;
//	CORCONbits.SATA = 1; /* Enable saturation */
//	CORCONbits.SATDW = 1; 	/* Enable saturation on data space write */

//	dot_product(ADMIT_N_COEFF-1, admit_coeff, admit_state);

//	admit_state[1] = admit_state[0];
//	admit_state[0] = read_acca(0);
//	output = read_acca(variables->admit_shift);

//	__asm__ volatile ("disi #0x0000");
	
//	admit_state[3] = admit_state[2];

//	variables->RefPoint = output;
//}




	/* Z transorm shape of the admittance:
	 *    A + B z^-1 + C z^-2
	 * -------------------------
	 *    1 + D z^-1 + E z^-2 
	 */
/* We discretize if with the "Tustin" method */
void admit_update_coeff(SharedVariables* variables) {
	float a, b, c, d, e;
	float k = variables->admit_k / 64.;
	float f = variables->admit_d / 64.;
	float i = variables->admit_i / 64.;
	/* 500Hz */
	/* TODO: use ctrlooptime instead of the hardcoded value */
	float hcarre = 0.000004; 
	float h = 0.002;
	float denom;

	denom = k * hcarre + 2 * f * h + 4 * i;
	
	if(denom == 0) 
		return;

	/* we need to get the correct gain for this TF */
	a = (-0.004898 * (1<<13) / (2 * 3.14159265358979)) * hcarre / denom;
	b = 2 * a;
	c = a;

	d = -(2 * k * hcarre - 8 * i) / denom;

	e = -(k * hcarre - 2 * f * h + 4 * i)/denom;

	__asm__ volatile ("disi #0x3FFF");
	admit_coeff[0] = d;
	admit_coeff[1] = e;
	admit_coeff[2] = a;
	admit_coeff[3] = b;
	admit_coeff[4] = c;
	__asm__ volatile ("disi #0x0000");
}

void admit_reset_state(SharedVariables* variables) {
	int i = 0;

	__asm__ volatile ("disi #0x3FFF");
	for(; i < ADMIT_N_COEFF; i++) {
		admit_state[i] = 0;
		admit_state_arw[i] = 0;
	}
	__asm__ volatile ("disi #0x0000");
}

static int _do_admit_calc(int Torque, float * state) {
	float output = 0;
	int i;
	
	state[2] = Torque;
	for(i = 0; i < ADMIT_N_COEFF; i++) 
		output += state[i] * admit_coeff[i];
	
	state[1] = state[0];
	state[0] = output;

	state[4] = state[3];
	state[3] = state[2];

	return (int) output;
}

int do_admit_calc(SharedVariables* variables) {
	return _do_admit_calc(variables->Torque, admit_state);
}

int do_admit_calc_arw(SharedVariables* variables, int sat) {
	if(sat) 
		return _do_admit_calc(variables->Torque,admit_state_arw);

	return _do_admit_calc(0,admit_state_arw);
}
