//////////////////////////////////////////////////////////////////////
//   SwitchingModes.h
//////////////////////////////////////////////////////////////////////
//
//   Description: 
//   Multiple switching modes available for the motor. Hide hardware
//   implementation by offering higher level interfaces (generic) for users.
//
//   Author: Dominic Letourneau
//   Creation Date: 2006/11/06
//
//   Modification history:
//
//  Name					Date        Descrition
//	Dominic Letourneau		2006/11/06	Added brushless modes					
//  Dominic Letourneau		2006/11/30	Added brushed modes
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius / RoboMotio  2006
//////////////////////////////////////////////////////////////////////
#ifndef _SWITCHING_MODES_H_
#define _SWITCHING_MODES_H_

//Motor direction
#define CW	0		
#define CCW	1	

#ifndef NULL
#define NULL 0
#endif

/**
SWITCHING TABLE FOR BLDC MOTORS (60 electrical degrees)

-------PWM STATE ------ | ---CURRENT FLOW -- | --SECTOR-- | -HA- | -HB- | -HC- |     ^ 
g_BLDC_StateLoTable[0]  |       A-->B        |      0     |   0  |   0  |   1  |     |
g_BLDC_StateLoTable[1]  |       A-->C        |      1     |   0  |   1  |   1  |     |
g_BLDC_StateLoTable[2]  |       B-->C        |      2     |   0  |   1  |   0  |     |
g_BLDC_StateLoTable[3]  |       B-->A        |      3     |   1  |   1  |   0  |     |
g_BLDC_StateLoTable[4]  |       C-->A        |      4     |   1  |   0  |   0  |     | 
g_BLDC_StateLoTable[5]  |       C-->B        |      5     |   1  |   0  |   1  |    CCW

Notes:

1) Only one bit at a time will change from one sector to another (grey code).
2) [Sector + 3] % 6 ---> Current flows in the opposite direction.
3) Refer to section "Motor Control PWM" for hardware documentation of dsPIC30FXXXX.

**/

//TODO Add Brushed motor modes
//Switching modes
typedef enum {
	BLDC_NSR_FD,//Brushless DC motor, non sync. fast decay mode
	BLDC_SR_SD, //Brushless DC motor, sync. rect., slow decay
	BLDC_NSR_SD, //Brushless DC motor, non sync., slow decay
	BLDC_SR_SD_SINUS, //Brushless DC motor, sync. slow decay, Sinus-like
	BDC_NSR_FD,//Brushed DC motor, non sync. fast decay mode
	BDC_SR_SD, //Brushed DC motor, sync. rect., slow decay
	BDC_NSR_SD, //Brushed DC motor, non sync., slow decay
	BDC_SR_SD_SINUS //Brushed DC motor, sync. slow decay, Sinus-like
} SwitchingMode;

/**
	Init the motor switching mode and PWM frequency.
	\param mode The switching mode
	\param frequency The switching frequency in Hz
	\return void
*/
void InitSwitchingMode(SwitchingMode mode, unsigned int frequency);

/**
	Set The PWM duty cycle.
	\param duty Maximum value is depending on the PTPER register [-PTPER,PTPER]
	\return void
*/
void setPWMDuty(signed int duty, unsigned char idle_enabled);

/**
	Get the actual PWM duty cycle.
	\param none
	\return signed int, the current duty cycle
*/
signed int getPWMDuty(void);

/**
	Get the current motor direction
	\param none
	\return unsigned char, the current direction (CW,CCW)
*/
unsigned char getDirection(void);

/**
	Get the hall switching counter
	\param none
	\return signed int, the counter value
*/
signed int getHallSwitchingCounter();

/**
	Reset the hall switching counter to 0
	\param none
	\return void
*/
void resetHallSwitchingCounter();

/**
	Get Maximum duty cycle value
	\param none
	\return Maximu duty cycle value
*/
signed int getMaxDuty();


/**
	Get Minimum duty cycle value
	\param none
	\return Minimum duty cycle value
*/
signed int getMinDuty();

/**
	Get the A3935 State (!FAULT,UVFLT,OVFLT bits)
	\param none	
	\return unsigned char, !FAULT = bit 0, UVFLT = bit 1, OVFLT = bit2
*/
unsigned char getSwitchingStatus();


#endif
