//////////////////////////////////////////////////////////////////////
//  control.c
//////////////////////////////////////////////////////////////////////
//
//  Description: 
//  Control modes and speed profilesF
//
//  Author: Dominic Letourneau
//  Creation Date: 2007/01/30
//
//  Modification history:
//
//  Name					Date        Descrition
//  Dominic Letourneau		2009/02/11  Added new DADE control mode (impedance)
//  Dominic Letourneau		2007/09/12	Speed and Position ramping removed. The static variable caused invalid commands.
//	Dominic Letourneau		2007/09/07	Careful attention to MIN, MAX, SIGN, ABS. Moved to utils.h
//	Dominic Letourneau		2007/07/03	Added ramping in speed and position
//	Dominic Letourneau		2007/06/04	Added reset_pid to reset history in PID values when ERROR OR CONFIG modes.
//  Dominic Letourneau		2007/04/27	Validating software setPoint limits (OK)
//	Dominic Letourneau		2007/02/08	Avoid sending PWM commands when MODE = CONFIG
//	Dominic Letourneau		2007/02/05	Added control modes and types
//	Dominic Letourneau		2007/01/30	Implementation & tests of trapezoidal speed profile
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius / RoboMotio  2006,2007
//////////////////////////////////////////////////////////////////////

/*               *|
d/dt            * |
C|             *  |
T| 	          *   |
R|			 *    |
L|          *************************         -----------Speed MAX
 |         *      |                   *
V|        *       |                    *
A|       *        |                     *-------- Slope (m) = acceleration step
R|      *         |                      *      |
I|     *          |                       *     |
A|    *           |                        *    |  y = (m) x
B|   *            |                         *   |
L|  *             |                          *  |
E| *              |                           * |
-|*---------------|----------------------------*--------->
 Vinit(0)		  Vnext(Xp=Actual pos)			Vdest   Time
  -             CONTROL VARIABLE               -

The trick is simple : 

1) The area under the trapezoidal speed profile is the position change
2) Calculate the speed according to the function speed = acceleration step (m) * x
3) Take the minimum speed between the speed from the beginning, speed from the end and speed max
4) Next position to acheive is the integration of calculated speed over time until we reach destination

First slope :

Note : SLOPE2 = SLOPE 1 EVALUATED AT Vdest - Vnext

Speed(x) = Acceleration Step * (x)
Position(x) = (Acceleration Step * (x)^2) / 2 -------> Integration of speed

---> DeltaPosition = (acceleration step / 2) * [(x)^2)] : Integral from 0 to Position Xp = Vnext - Vinit
---> (Vnext - Vinit) = (acceleration step / 2) * [(Vnext - Vinit)^2]
---> (2 *(Vnext - Vinit)) / (acceleration_step) = (Vnext - Vinit)^2
---> sqrt(2*(Vnext - Vinit) / acceleration_step) = (Vnext - Vinit)
---> sqrt(2*(Vnext - Vinit) / acceleration_step) * acceleration_step = Speed(Vnext -Vinit)
	 
FINAL RESULT FOR SPEED CALCULATION :
	 sqrt(2*(Vnext - Vinit) * acceleration_step) = Speed(Vnext -Vinit)


FOR TRAPEZOIDAL SPEED PROFILE ---

speed_from_begin = sqrt(2*(Vnext - Vinit) * acceleration_step)
speed_from end = sqrt(2*(Vdest - Vnext) * acceleration_step)

SELECTED SPEED = min(speed_from_begin,speed_from_end,speed_limit)

Note :

IF speed limit is very high or IF we do not have the time to reach the speed limit
THEN we will get a triangular shape for the speed profile

*/


#include "control.h"
#include "PID.h"
#include "Device.h"
#include "PWM.h"
#include "CANShared.h"
#include "PWM.h"
#include "utils.h"
#include "main.h"
#include "sensors.h"



//This is hidden from the user to avoid problems.
void control_trapezoidal_profile(SharedVariables* variables, MotorId id);
void control_open_loop(SharedVariables* variables, MotorId id);
void control_position(SharedVariables* variables, MotorId id);
void control_speed(SharedVariables* variables, MotorId id);


/* Integer square root by Halleck's method, with Legalize's speedup */
long isqrt (long x) 
{
  long  squaredbit, remainder, root;

   if (x<1) return 0;
  
   /* Load the binary constant 01 00 00 ... 00, where the number
    * of zero bits to the right of the single one bit
    * is even, and the one bit is as far left as is consistant
    * with that condition.)
    */
   squaredbit  = (long) ((((unsigned long) ~0) >> 1) & 
                        ~(((unsigned long) ~0) >> 2));
   /* This portable load replaces the loop that used to be 
    * here, and was donated by  legalize@xmission.com 
    */

   /* Form bits of the answer. */
   remainder = x;  root = 0;
   while (squaredbit > 0) {
     if (remainder >= (squaredbit | root)) {
         remainder -= (squaredbit | root);
         root >>= 1; root |= squaredbit;
     } else {
         root >>= 1;
     }
     squaredbit >>= 2; 
   }

   return root;
}


/*
	This will do the appropriate job according to control type and current mode.

*/
void control_main_handle(SharedVariables* variables, MotorId id)
{
	
	//WARNING :
	//WE SUPPOSE HERE THAT THE SPEED, POSITION AND TORQUE VARIABLES
	//ARE UP TO DATE!

	//LIMIT SETPOINT IF REQUIRED
	variables->SetPoint = MAX(variables->SetPointMin,MIN(variables->SetPoint,variables->SetPointMax));


	switch(variables->CtrlMode)
	{
		case CTRL_MODE_CONFIG:
			//Make sure PID Output is 0
			//No PWM will be applied
			variables->PIDOut = 0;

			//SAFETY -  Reset control history
			pid_reset(variables);

			//TODO : Check valid configuration
		break;

		
		case CTRL_MODE_NORMAL:

			if (variables->ThermalState > MAX_THERMAL_COUNT_CYCLES)
			{
				//Thermal switch activated, overheating
				//Stop motor
				variables->PIDOut = 0;

				//Set error code
				variables->ErrorCode |= ERROR_MOTOR_OVERHEAT;

				//Reset PID
				pid_reset(variables);

				//Change control mode to error...
				variables->CtrlMode = CTRL_MODE_ERROR;
				
			}
			else
			{
				switch(variables->CtrlType)
				{
					case CTRL_TYPE_OPEN_LOOP:
						control_open_loop(variables,id);
					break;
		
					case CTRL_TYPE_POSITION:
						control_position(variables,id);						
					break;
			
					case CTRL_TYPE_POSITION_TRAPZ_PROFILE:
						control_trapezoidal_profile(variables,id);	
					break;
	
					case CTRL_TYPE_SPEED:
						control_speed(variables,id);
					break;
	
					default:
						//PUT BACK IN OPEN LOOP TYPE AND CONFIG MODE
						variables->CtrlType = CTRL_TYPE_OPEN_LOOP;
						variables->CtrlMode = CTRL_MODE_CONFIG;
					break;
				}
			}
		break;

		case CTRL_MODE_ERROR:

			//Make sure PID Output is 0
			//No PWM will be applied
			variables->PIDOut = 0;
			//SAFETY -  Reset control history
			pid_reset(variables);
		break;


		case CTRL_MODE_RESET:
			Reset();
		break;

		default:
			//Unknown mode, put back to config mode
			variables->CtrlMode = CTRL_MODE_CONFIG;
		break;
	}
}




/*
	This will output RefPoint (velocity) for the desired speed to be acheived (for position control).
	The profile of the velocity will be trapezoidal.
	
	Modified variables :
	
	SetPoint ->Velocity Command
	NextPoint ->Next position to acheive
	InitPoint ->Initial position
	DestPoint ->Destination position
*/
void control_trapezoidal_profile(SharedVariables* variables, MotorId id)
{

	signed int speed_from_begin = 0;
	signed int speed_from_end = 0;
	signed int speed_sign = 1;

	//UPDATE DESTINATION
	//SETPOINT CHANGED ?
	if (variables->DestPoint != variables->SetPoint)
	{
		//Re-initialize everyting
		variables->DestPoint = variables->SetPoint;
		variables->InitPoint = variables->Position;
		variables->NextPoint = variables->InitPoint;
	}
	

	//initialize speed sign
	if (variables->DestPoint < variables->Position)
	{
		speed_sign = -1;
	}	
	
	//CALCULATE SPEED FROM BEGIN
	//ADDING 1 TO AVOID HAVING A MINIMUM OF 0 WHEN STARTING (next_point = init_point)
	//OTHERWISE IT WOULD GIVE A SPEED OF 0 AND NOTHING WOULD MOVE		
	speed_from_begin = isqrt((long)(variables->AccelerationStep) * (1L + 2L * ABS(((long)variables->Position - (long)variables->InitPoint))));	


	//Make sure speed_from_begin is big enough to start movement
	speed_from_begin = MAX(5,speed_from_begin);

	//CALCULATE SPEED FROM END
	speed_from_end = isqrt((long)(variables->AccelerationStep) * (2L * ABS(((long)variables->DestPoint - (long)variables->Position))));
		
	//NEXT SPEED CALCULATION AS INPUT TO PID
	//SPEED = min(speed_from_begin,speed_from_end,speed_limit)
	variables->RefPoint = (speed_sign * MIN(MIN(speed_from_begin,speed_from_end),variables->SpeedMax));		
		
	//Position accumulator = Speed Integration
	variables->NextPoint = variables->Position + variables->RefPoint;
		
	//AVOID OVERSHOOT
	if (speed_sign < 0)
	{
		if (variables->NextPoint < variables->DestPoint)
		{
			variables->NextPoint = variables->DestPoint;
		}
	}
	else
	{
		if (variables->NextPoint > variables->DestPoint)
		{
			variables->NextPoint = variables->DestPoint;
		}
	}


	//UPDATE PID INPUTS (ACTUAL POSITION)
	//THIS IS FOR POSITION CONTROL...
	//variables->MesPoint = variables->Position;
	//variables->RefPoint = variables->NextPoint;


	//THIS IS FOR SPEED CONTROL
	//Actual control is speed control with profile
	//variables->RefPoint is calculated above...
	variables->MesPoint = variables->Speed;

	
	pid_calculate(variables);	
}

void control_open_loop(SharedVariables* variables, MotorId id)
{
	//SetPoint is the PWM Output...
	variables->PIDOut = variables->SetPoint;

	//NO PID REQUIRED!
	//Safety ...
	variables->RefPoint = 0;
	variables->MesPoint = 0;
}


void control_position(SharedVariables* variables, MotorId id)
{
	/*

	//Make sure acceleration is adequate and positive
	signed int absSpeed = ABS(variables->SpeedMax);
	signed int SpeedMax = MAX(1,absSpeed);

	//RAMPING TO DESIRED POSITION ACCORDING TO SPEED MAX
	//Update RefPoint
	if (variables->SetPoint >= (signed int) variables->Position)
	{
		variables->RefPoint = MIN(variables->SetPoint, (signed int) variables->Position + SpeedMax);
	}
	else
	{
		variables->RefPoint = MAX(variables->SetPoint, (signed int) variables->Position - SpeedMax);
	}


	//DIRECT POSITION COMMAND (USE THIS IF NO RAMPING REQUIRED)
	//variables->RefPoint = variables->SetPoint;


	//Update MesPoint
	variables->MesPoint = variables->Position;

	pid_calculate(variables);

	*/


	signed int speed_from_begin = 0;
	signed int speed_from_end = 0;
	signed int speed_sign = 1;

	//UPDATE DESTINATION
	//SETPOINT CHANGED ?
	if (variables->DestPoint != variables->SetPoint)
	{
		//Re-initialize everyting
		variables->DestPoint = variables->SetPoint;
		variables->InitPoint = variables->Position;
		variables->NextPoint = variables->InitPoint;
	}
	

	//initialize speed sign
	if (variables->DestPoint < variables->Position)
	{
		speed_sign = -1;
	}	
	
	//CALCULATE SPEED FROM BEGIN
	//ADDING 1 TO AVOID HAVING A MINIMUM OF 0 WHEN STARTING (next_point = init_point)
	//OTHERWISE IT WOULD GIVE A SPEED OF 0 AND NOTHING WOULD MOVE		
	speed_from_begin = isqrt((long)(variables->AccelerationStep) * (1L + 2L * ABS(((long)variables->NextPoint - (long)variables->InitPoint))));	

	//CALCULATE SPEED FROM END
	speed_from_end = isqrt((long)(variables->AccelerationStep) * (2L * ABS(((long)variables->DestPoint - (long)variables->NextPoint))));
		
	//NEXT SPEED CALCULATION AS INPUT TO PID
	//SPEED = min(speed_from_begin,speed_from_end,speed_limit)
	variables->RefPoint = (speed_sign * MIN(MIN(speed_from_begin,speed_from_end),variables->SpeedMax));		
		
	//Position accumulator = Speed Integration
	variables->NextPoint += variables->RefPoint;
		
	//AVOID OVERSHOOT
	if (speed_sign < 0)
	{
		if (variables->NextPoint < variables->DestPoint)
		{
			variables->NextPoint = variables->DestPoint;
		}
	}
	else
	{
		if (variables->NextPoint > variables->DestPoint)
		{
			variables->NextPoint = variables->DestPoint;
		}
	}


	//UPDATE PID INPUTS (ACTUAL POSITION)
	//THIS IS FOR POSITION CONTROL...
	variables->MesPoint = variables->Position;
	variables->RefPoint = variables->NextPoint;
	
	pid_calculate(variables);	

}

void control_speed(SharedVariables* variables, MotorId id)
{



	//Make sure acceleration is adequate and positive
	signed int absAccStep = ABS(variables->AccelerationStep);
	signed int acceleration = MAX(1,absAccStep);
	
	//RAMPING TO DESIRED SPEED ACCORDING TO ACCELERATION
	//Update RefPoint and MesPoint

	if (variables->SetPoint >= variables->Speed)
	{
	
		variables->RefPoint = MIN(variables->SetPoint, variables->RefPoint + acceleration);
		
		if (variables->RefPoint < variables->Speed)
		{
			variables->RefPoint = variables->Speed;
		}
	
	}
	else
	{
		variables->RefPoint = MAX(variables->SetPoint, variables->RefPoint - acceleration);
	
		if (variables->RefPoint > variables->Speed)
		{
			variables->RefPoint = variables->Speed;
		}
	
	}


	//DIRECT SPEED COMMAND (USE THIS IF NO RAMPING REQUIRED)
	//variables->RefPoint = variables->SetPoint;

	//Update actual speed
	variables->MesPoint = variables->Speed;

	pid_calculate(variables);
}


