
//////////////////////////////////////////////////////////////////////
//  currentLimit.c
//////////////////////////////////////////////////////////////////////
//
//  Description: 
//  Software current limiting for motor protection
//
//  Author: Dominic Letourneau
//  Creation Date: 2007/02/08
//
//  Modification history:
//
//  Name					Date        Descrition
//	Dominic Letourneau		2007/06/19	Avoid changing current limit state when setpoint changes
//	Dominic Letourneau		2007/02/08	Making sure we use absolute values for current reading and limit
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius / RoboMotio  2006,2007
//////////////////////////////////////////////////////////////////////

#include "currentLimit.h"
#include "switchingModes.h"
#include "Device.h"
#include "CANShared.h"
#include "utils.h"


/*
	Software current limit that allows to set a PWM correction step and
	a maximum current.

	Modified variables :
	
	PWM_CurrentLimit

	Inputs :

	signed int CurrentLimit; //unit = mA
	controlOutputMax --- PWM Max
	controlOutputMin --- PWM Min
	PWM_CurrentStep --- PWM Step to be applied for correction if over current

*/
void current_software_limit(SharedVariables* variables)
{

	signed int min_duty = getMinDuty();
	signed int max_duty = getMaxDuty();
	int pwmStep = 1;
	
	if (variables->PWM_CurrentStep != 0)
	{
		pwmStep = ABS(variables->PWM_CurrentStep);
	}

	//Current read by the module will always be positive
	if (ABS(variables->Current) < ABS(variables->CurrentLimit))
	{
		if (variables->CurrentLimitActive)
		{
			//SLOWLY INCREMENTING PWM OUTPUT
			if (variables->PIDOut < 0)
			{
				variables->PWM_CurrentLimit = MAX(variables->PIDOut, MAX(variables->PWM_CurrentLimit - pwmStep,min_duty));
				
				//Verify if we still need to limit the current
				if (variables->PIDOut >= variables->PWM_CurrentLimit)
				{
					variables->CurrentLimitActive = 0;
				}
			}
			else
			{
				variables->PWM_CurrentLimit = MIN(variables->PIDOut, MIN(variables->PWM_CurrentLimit + pwmStep,max_duty));

				//Verify if we still need to limit the current
				if (variables->PIDOut <= variables->PWM_CurrentLimit)
				{
					variables->CurrentLimitActive = 0;
				}
			}


			//TODO : Empty PID Integral Accum ?
			//Right now this function will reset all error history, but accumulator is not cleared
			//pid_reset(variables);
		}
		else
		{
			//NO LIMIT APPLIED, DIRECT OUTPUT
			variables->PWM_CurrentLimit = variables->PIDOut;
		}
	}
	else 
	{	
		//THIS WILL LOWER THE PWM LIMIT UNTIL WE REACH THE CURRENT THRESHOLD (OR 0 PWM)
		if (variables->PIDOut < 0)
		{
			variables->PWM_CurrentLimit = MIN(0,MAX(variables->PWM_CurrentLimit + pwmStep,variables->PIDOut));
		}
		else
		{
			variables->PWM_CurrentLimit = MAX(0,MIN(variables->PWM_CurrentLimit - pwmStep,variables->PIDOut));
		}

		variables->CurrentLimitActive = 1;
	}

	//Current limit will be applied in the main loop
}


