
//////////////////////////////////////////////////////////////////////
// main.c
//////////////////////////////////////////////////////////////////////
//
//	Description: 
//	Main motor control program.
//
//	Author: Dominic Letourneau
//	Creation Date: 2006/11/06
//
//	Modification history:
//
//	Name					Date        Descrition
//  Julien Fremy			2008/10/16	Correction of a bug due to the discontinuity of the position sensor 
//										in the speed measure
//  Dominic Letourneau		2008/10/09	Added speed measure with the derivative of the position
//	Pierre Lepage				2008/05/14	Added sensor_read_input_capture_high() and _low() in read_sensor()
//	Dominic Letourneau		2008/02/04	Added (empty) call for 16bits ADC
//	Dominic Letourneau		2008/02/04	Added setpoint with torque correction
//	Dominic	Letourneau		2008/01/08	Do not clear error code in error mode
//	Dominic Letourneau		2008/01/08	Added protection to avoid changing from error mode to normal mode
//	Dominic Letourneau		2008/01/07	Added Torque limit & watchdog timer (TIMER4)
//	Dominic Letourneau		2007/07/17	Added initialization of default values in case of new module detected.
//  Dominic Letourneau		2007/06/05	Cleanup Code, removed timer2, added feedback flag for periodic output, moved CAN 
//										transmission in the main loop instead of interrupts. Added FeedbackScaler.
//	Dominic Letourneau		2007/06/04	Going back to CONFIG mode when E-STOP detected
//  Dominic Letourneau		2007/05/31	Added Startup mode from EEPROM
//	Dominic Letourneau		2007/05/28	Added STATE1 & STATE2 CAN PERIODIC OUTPUT
//	Dominic Letourneau		2007/05/28	Added CAN Emergency and other functions.
//	Dominic Letourneau		2007/05/28	Added all required sensors measure inputs
//	Dominic Letourneau		2007/05/10	Added ErrorCode
//	Dominic Letourneau		2007/05/10	Live change in switching modes
//	Dominic Letourneau		2007/05/09	Added Profiles
//	Dominic Letourneau		2007/05/09	Added limit switches and A3935 status
//	Dominic	Letourneau		2007/05/04	Added FeedbackTimeBase
//	Dominic Letourneau		2007/05/03	Added setpoint by CAN, AN1 or AN2. Dead zone for ANALOG around setpoint = 0
//  Dominic Letourneau		2007/04/27  Update E-Stop state
//  Dominic	Letourneau		2007/03/29	Added EXT1 and EXT2 sensors
//	Dominic Letourneau		2007/02/08	Added maximum temperature and error codes in CTRLType
//	Dominic Letourneau		2007/02/08	Must be in CTRL_MODE_CONFIG to save to eeprom
//	Dominic Letourneau		2007/02/08	Added CTRL_MODE_RESET & init test
//	Dominic Letourneau		2007/02/08	Added Current, ADC1 & ADC2 offsets
//	Dominic	Letourneau		2007/02/08	Added motor switching mode
//	Dominic Letourneau		2007/02/08	Added current sign according to PWM duty
//	Dominic Letourneau		2007/02/08	Added motor and encoder bias
//  	
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius / RoboMotio  2007
//////////////////////////////////////////////////////////////////////

#include "Device.h"
#include "main.h"
#include <stdio.h>
#include <uart.h>
#include <dsp.h>
#include <timer.h>
#include <stdlib.h>
#include <string.h>
#include "PID.h"
#include "control.h"
#include "sensors.h"
#include "CANDriver.h"
#include "CANShared.h"
#include "currentLimit.h"
#include "memory.h"
#include "utils.h"
#include "RST-imped.h"
#include "PWM.h"

//--------------------------Device Configuration------------------------
//_FOSC(CSW_FSCM_OFF & HS2_PLL8);
//_FWDT(WDT_OFF);
//_FBORPOR(PBOR_OFF & BORV_42 & PWRT_64 & MCLR_EN);
//----------------------------------------------------------------------


void read_sensors();
void do_can_feedback();

unsigned char doCANFeedback = 0;

void InitTMR1(void);	// Initialization for TIMER1
void InitTMR4(void);	// Initialization for TIMER3



/**********************************************************************
	Timer 1 interrupt.

	Slow sensor update and led flashing.

**********************************************************************/
/*
const unsigned int SINE_WAVE[128] = {
0,2,5,7,10,12,15,17,20,22,24,27,29,32,34,36,39,41,43,45,47,50,52,54,56,58,60,62,64,66,68,69,71,73,75,76,78,79,81,82,84,85,86,87,89,90,91,92,93,94,94,
95,96,97,97,98,98,99,99,99,100,100,100,100,100,100,100,100,99,99,99,98,98,97,97,96,95,94,94,93,92,91,90,89,87,86,85,84,82,81,79,78,76,75,73,71,69,68,
66,64,62,60,58,56,54,52,50,47,45,43,41,39,36,34,32,29,27,24,22,20,17,15,12,10,7,5,2,0};


unsigned int sine_index = 0;
unsigned int cycle_count = 0;

*/


void __attribute__((interrupt(__save__(CORCON)),auto_psv)) _T1Interrupt (void)
{
	IFS0bits.T1IF = 0;

	//LED FLASHING
	PORTGbits.RG13 = ~PORTGbits.RG13;


	doCANFeedback = 1;

	//SLOW REFRESHING SENSORS HERE...
    //__asm__ volatile ("disi #0x3FFF");
    //__asm__ volatile ("disi #0x0000");

	return;
}




/**********************************************************************
	init_default_variables()

	Init default variables

***********************************************************************/
void init_default_variables()
{
	unsigned int i = 0;
	
	//default everything to zero...
	//memset(g_sharedVariables,0,sizeof(SharedVariables) * 4);

	for (i = 0; i < 4; i++)
	{
	
		//POSITION MODE
		g_globalVariables.m_sharedVariables[i].CtrlType = CTRL_TYPE_POSITION;
		g_globalVariables.m_sharedVariables[i].CtrlMode = CTRL_MODE_NORMAL;
		
		g_globalVariables.m_sharedVariables[i].CurrentLimit = 1000;
		g_globalVariables.m_sharedVariables[i].SetPointMax = 4096;
		g_globalVariables.m_sharedVariables[i].SetPointMin = 0;
		g_globalVariables.m_sharedVariables[i].MotorBias = 0;
		g_globalVariables.m_sharedVariables[i].PIDOut = 0;
		g_globalVariables.m_sharedVariables[i].SpeedMax = 10;
		g_globalVariables.m_sharedVariables[i].AccelerationStep = 1;
		g_globalVariables.m_sharedVariables[i].SetPointSource = SETPTSRC_CAN;
		g_globalVariables.m_sharedVariables[i].pid_error_accum_max = 1000.0;
		g_globalVariables.m_sharedVariables[i].pid_kp = 0;
		g_globalVariables.m_sharedVariables[i].pid_ki = 0;
		g_globalVariables.m_sharedVariables[i].pid_kd = 0;

		#if 0
		//OPEN LOOP DEFAULT
		g_globalVariables.m_sharedVariables[i].CtrlType = CTRL_TYPE_OPEN_LOOP;
		g_globalVariables.m_sharedVariables[i].CtrlMode = CTRL_MODE_NORMAL;
		
		g_globalVariables.m_sharedVariables[i].CurrentLimit = 1000;
		g_globalVariables.m_sharedVariables[i].SetPointMax = getMaxDuty();
		g_globalVariables.m_sharedVariables[i].SetPointMin = getMinDuty();
		g_globalVariables.m_sharedVariables[i].MotorBias = 0;
		g_globalVariables.m_sharedVariables[i].PIDOut = 0;
		#endif
	
	}
	

	//Update EEPROM
	WriteGlobalVariables(&g_globalVariables);
	
		
	//write default profile to 0
	can_write_eeprom(CAN_EEPROM_STARTUP_PROFILE_ADDRESS,0);

	//write default mode to config
	can_write_eeprom(CAN_EEPROM_STARTUP_MODE_ADDRESS, CTRL_MODE_NORMAL);
	
	
}

/**********************************************************************
	void read_sensors()

	Read all sensors...

***********************************************************************/
void read_sensors()
{

	int i =0;
	signed long old_pos = 0;
	unsigned int old_speed = 0;
	int adcvalue;


	for (i = 0; i < 4; i++)
	{
		SharedVariables *variables = &g_globalVariables.m_sharedVariables[i];

		adcvalue = sensors_read_adc(i);
	
		//adc valid	
		if (adcvalue < 0)
		{
			variables->CtrlMode = CTRL_MODE_ERROR;
			variables->ErrorCode |= ERROR_ADC_INVALID;
			variables->PIDOut = 0;
		}
	

		if(variables->EncoderBias)
		{
			//Update input capture
			variables->ICValue = -1 * (variables->ICOffset + sensors_read_input_capture(i));

			//Update adc
			variables->ADCValue = -1 * (variables->ADCOffset + adcvalue);
		}
		else
		{
		
			//Update input capture
			variables->ICValue = variables->ICOffset + sensors_read_input_capture(i);
		
			//Update adc
			variables->ADCValue = variables->ADCOffset + adcvalue;	
		}	


		//Update current
		variables->Current = variables->CurrentOffset + sensors_read_current(i) * SIGN(getPWMDuty(i));
		
		//Update thermal state
		variables->ThermalState = sensors_read_thermal_state(i);

		//Store old value for derivatives
		old_pos = variables->Position;
		old_speed = variables->Speed;
	
		//LOOK FOR E-STOP
		g_globalVariables.m_ESTOPEnabled = !LATFbits.LATF6;


		//Update Position Variable
		switch(variables->PosMesSource)
		{
			case MS_ANALOG:
				//DIRECT ANALOG
				variables->Position = variables->ADCValue;		
			break;

			case MS_INPUT_CAPTURE:
				//Input Capture
				variables->Position = variables->ICValue;
			break;

			default:
				//DIRECT ANALOG
				variables->Position = variables->ADCValue;	
			break;
		}

		//Update Speed Variable
		//Warning - We are not multi turn
		variables->Speed = variables->Position - old_pos;

		//Update Acceleration
		variables->Acceleration = variables->Speed - old_speed;

	}

	return;
}


/**********************************************************************
	Timer 4 interrupt.

	This is the main control loop

***********************************************************************/
//LABORIUS_MESSAGE msg;

void __attribute__((interrupt(__save__(CORCON)),auto_psv)) _T4Interrupt (void)
{
	int i = 0;

	//Clear interrupt flag
	IFS1bits.T4IF = 0;

	//Read all sensors first
	read_sensors();

	//Control loop for all motors
	for (i = 0; i < 4; i++)
	{
		//SetPoint handling (dead zone)
		switch(g_globalVariables.m_sharedVariables[i].SetPointSource)
		{
			case SETPTSRC_CAN:
				//NOTHING TO DO, SETPOINT WILL BE SET BY
				//UPDATING CAN VARIABLES (PROC_MESSAGE OR CAN DRIVER)
			break;
		
			case SETPTSRC_AN1:
				g_globalVariables.m_sharedVariables[i].SetPoint = g_globalVariables.m_sharedVariables[MOTOR_1].ADCValue;
			break;
		
			case SETPTSRC_AN2:
				g_globalVariables.m_sharedVariables[i].SetPoint = g_globalVariables.m_sharedVariables[MOTOR_2].ADCValue;
			break;
	
			case SETPTSRC_AN3:
				g_globalVariables.m_sharedVariables[i].SetPoint = g_globalVariables.m_sharedVariables[MOTOR_3].ADCValue;
			break;
	
			case SETPTSRC_AN4:
				g_globalVariables.m_sharedVariables[i].SetPoint = g_globalVariables.m_sharedVariables[MOTOR_4].ADCValue;
			break;
		
			default:
			//DO NOTHING
			break;
		}


		//CALL CONTROL HANDLE
		control_main_handle(&g_globalVariables.m_sharedVariables[i],i);

		//Software current limit
		//This will update PWM_CurrentLimit value to be applied
		//This must be the last function call before applying PWM
		current_software_limit(&g_globalVariables.m_sharedVariables[i]);

		//THIS WILL BYPASS CURRENT LIMIT
		//g_globalVariables.m_sharedVariables[i].PWM_CurrentLimit = g_globalVariables.m_sharedVariables[i].PIDOut;

		//Set PWM Value according to motor bias	
		if (g_globalVariables.m_sharedVariables[i].MotorBias)
		{
			setPWMDuty(i, -g_globalVariables.m_sharedVariables[i].PWM_CurrentLimit);
		}
		else
		{
			setPWMDuty(i, g_globalVariables.m_sharedVariables[i].PWM_CurrentLimit);
		}
	}


	//Update timebase (live!)
	//PR4 = g_sharedVariables->CtrlTimeBase;
	g_globalVariables.m_loopTime = TMR4;


	return;
}

/************************************************************************
void proc_message(LABORIUS_MESSAGE *message)

CAN MESSAGES HANDLING HERE!

**************************************************************************/
void proc_message(LABORIUS_MESSAGE *message)
{
	int i = 0;
	BootConfig *bootConfig = can_get_boot_config();

	if (bootConfig)
	{
		if (message->msg_dest == bootConfig->module_id)
		{
			if (message->msg_type == CAN_TYPE_EMERGENCY)
			{
				switch (message->msg_cmd)
				{
					case CAN_EMERGENCY_CMD_STOP:
						//SWITCH BACK TO CONFIG MODE
						for (i = 0; i < 4; i++)
						{
							g_globalVariables.m_sharedVariables[i].CtrlMode = CTRL_MODE_CONFIG;
						}
					break;

					case CAN_EMERGENCY_CMD_RESET:
						Reset();
					break;

					case CAN_EMERGENCY_CMD_PROGRAM:
						//CAN WE DO BETTER?
						bootConfig->module_state = BOOT_IDLE;
						can_write_boot_config(bootConfig);
						Reset();
					break;
				}
			}//TYPE = EMERGENCY	
			else if (message->msg_type == CAN_TYPE_ACTUATOR_HIGH_PRIORITY)
			{

				switch(message->msg_cmd)
				{
					case CAN_ACTUATOR_CMD_POSITIONS:

						message->msg_data[0] = g_globalVariables.m_sharedVariables[MOTOR_1].Position;
						message->msg_data[1] = g_globalVariables.m_sharedVariables[MOTOR_1].Position >> 8;
						message->msg_data[2] = g_globalVariables.m_sharedVariables[MOTOR_2].Position;
						message->msg_data[3] = g_globalVariables.m_sharedVariables[MOTOR_2].Position >> 8;
						message->msg_data[4] = g_globalVariables.m_sharedVariables[MOTOR_3].Position;
						message->msg_data[5] = g_globalVariables.m_sharedVariables[MOTOR_3].Position >> 8;
						message->msg_data[6] = g_globalVariables.m_sharedVariables[MOTOR_4].Position;
						message->msg_data[7] = g_globalVariables.m_sharedVariables[MOTOR_4].Position >> 8;
						//make sure we are using the right length and remove the remote flag
						message->msg_data_length = 8;
						message->msg_remote = 0;

						//send the answer
						while(can_send_laborius_packet(message));

					break;

					case CAN_ACTUATOR_CMD_SPEEDS:

						message->msg_data[0] = g_globalVariables.m_sharedVariables[MOTOR_1].Speed;
						message->msg_data[1] = g_globalVariables.m_sharedVariables[MOTOR_1].Speed >> 8;
						message->msg_data[2] = g_globalVariables.m_sharedVariables[MOTOR_2].Speed;
						message->msg_data[3] = g_globalVariables.m_sharedVariables[MOTOR_2].Speed >> 8;
						message->msg_data[4] = g_globalVariables.m_sharedVariables[MOTOR_3].Speed;
						message->msg_data[5] = g_globalVariables.m_sharedVariables[MOTOR_3].Speed >> 8;
						message->msg_data[6] = g_globalVariables.m_sharedVariables[MOTOR_4].Speed;
						message->msg_data[7] = g_globalVariables.m_sharedVariables[MOTOR_4].Speed >> 8;
						//make sure we are using the right length and remove the remote flag
						message->msg_data_length = 8;
						message->msg_remote = 0;

						//send the answer
						while(can_send_laborius_packet(message));
					break;

					case CAN_ACTUATOR_CMD_SETPOINTS:

						message->msg_data[0] = g_globalVariables.m_sharedVariables[MOTOR_1].SetPoint;
						message->msg_data[1] = g_globalVariables.m_sharedVariables[MOTOR_1].SetPoint >> 8;
						message->msg_data[2] = g_globalVariables.m_sharedVariables[MOTOR_2].SetPoint;
						message->msg_data[3] = g_globalVariables.m_sharedVariables[MOTOR_2].SetPoint >> 8;
						message->msg_data[4] = g_globalVariables.m_sharedVariables[MOTOR_3].SetPoint;
						message->msg_data[5] = g_globalVariables.m_sharedVariables[MOTOR_3].SetPoint >> 8;
						message->msg_data[6] = g_globalVariables.m_sharedVariables[MOTOR_4].SetPoint;
						message->msg_data[7] = g_globalVariables.m_sharedVariables[MOTOR_4].SetPoint >> 8;
						//make sure we are using the right length and remove the remote flag
						message->msg_data_length = 8;
						message->msg_remote = 0;

						//send the answer
						while(can_send_laborius_packet(message));
					break;


					case CAN_ACTUATOR_CMD_WRITE_SETPOINT:

						if (message->msg_data_length == 3)
						{
							//First byte is the drive id, then the setpoint LSB, MSB
							//Making sure we are handling data correctly
							switch(message->msg_data[0])
							{
								case MOTOR_1:
									__asm__ volatile ("disi #0x3FFF");
									g_globalVariables.m_sharedVariables[MOTOR_1].SetPoint = message->msg_data[1];
									g_globalVariables.m_sharedVariables[MOTOR_1].SetPoint |= (((int)message->msg_data[2]) << 8);
									__asm__ volatile ("disi #0x0000");
								break;

								case MOTOR_2:
									__asm__ volatile ("disi #0x3FFF");
									g_globalVariables.m_sharedVariables[MOTOR_2].SetPoint = message->msg_data[1];
									g_globalVariables.m_sharedVariables[MOTOR_2].SetPoint |= (((int)message->msg_data[2]) << 8);
									__asm__ volatile ("disi #0x0000");
								break;

								case MOTOR_3:
									__asm__ volatile ("disi #0x3FFF");
									g_globalVariables.m_sharedVariables[MOTOR_3].SetPoint = message->msg_data[1];
									g_globalVariables.m_sharedVariables[MOTOR_3].SetPoint |= (((int)message->msg_data[2]) << 8);
									__asm__ volatile ("disi #0x0000");
								break;

								case MOTOR_4:
									__asm__ volatile ("disi #0x3FFF");
									g_globalVariables.m_sharedVariables[MOTOR_4].SetPoint = message->msg_data[1];
									g_globalVariables.m_sharedVariables[MOTOR_4].SetPoint |= (((int)message->msg_data[2]) << 8);
									__asm__ volatile ("disi #0x0000");
								break;

								default:
								break;
							}
			
							
						} 

					break;
				}//switch cmd
			}
		}
	}	
}

void safe_setpoint(SharedVariables *variables)
{

	//SAFETY, START WITH ACTUAL POSITION, SPEED = 0 OR TORQUE = 0 VALUES
	switch (variables->CtrlType)
	{
		
		case CTRL_TYPE_OPEN_LOOP:
			//START WITH 0 PWM
			variables->SetPoint = 0;
			variables->Speed =0;
		break;

		case CTRL_TYPE_POSITION:
			//START AT THE ACTUAL POSITION
			variables->InitPoint = variables->Position;
			variables->DestPoint = variables->Position;
			variables->NextPoint = variables->Position;
			variables->SetPoint = variables->Position;
			variables->Speed =0;
		break;

		case CTRL_TYPE_POSITION_TRAPZ_PROFILE:
			//START AT THE ACTUAL POSITION
			variables->InitPoint = variables->Position;
			variables->DestPoint = variables->Position;
			variables->NextPoint = variables->Position;
			variables->SetPoint = variables->Position;
			variables->Speed =0;
		break;

		case CTRL_TYPE_SPEED:
			//START WITH SPEED=0
			variables->SetPoint = 0;
			variables->Speed =0;
		break;

		default :
			variables->SetPoint = 0;
		break;
	}
	
}


/************************************************************************
	MAIN PROGRAM
**************************************************************************/
int main(void)
{	
	unsigned char i=0;
	LABORIUS_MASK mask_in[2];
	LABORIUS_FILTER filter_in[6];
	BootConfig *bootConfig = NULL;
	unsigned char Can_Addr = 0;
	//unsigned int shared_size = sizeof(SharedVariables);
	//unsigned int global_size = sizeof(GlobalVariables);
	
	//output G13 (LED)
	TRISGbits.TRISG13 = 0;
	PORTGbits.RG13 = 1;

	//reading boot config and device configuration
	//MUST BE DONE BEFORE INITIALIZING CAN MODULE
	bootConfig = can_get_boot_config();

	if (bootConfig)
	{
		//read configuration
		can_read_boot_config(bootConfig);
		
		//safety
		bootConfig->module_state = BOOT_NORMAL;
	
		//verify if we have correct configuration
		//write it back if not updated
		if (bootConfig->table_version !=  MODULE_TABLE_VERSION
		|| bootConfig->project_id != MODULE_PROJECT_ID
		|| bootConfig->code_version != MODULE_CODE_VERSION)
		{
			bootConfig->table_version = MODULE_TABLE_VERSION;
			bootConfig->project_id = MODULE_PROJECT_ID;
			bootConfig->code_version = MODULE_CODE_VERSION;

			//CHANGE THE MODULE ID FOR 
			//THE SECOND CONTROLLER...
			bootConfig->module_id = 0x01;

			can_write_boot_config(bootConfig);

			//THIS WILL UPDATE DEFAULT EEPROM CONFIGURATION FOR ALL MOTORS
			init_default_variables();
		}
	}
	
	//UPDATE CAN ADDRESS
	Can_Addr = bootConfig->module_id;
	
	// init mask
	for(i=0;i<2;i++){
		mask_in[i].mask_priority = 0;
		mask_in[i].mask_type = 0;
		mask_in[i].mask_cmd = 0;
		mask_in[i].mask_dest = 0xFF;
	}
	
	// init filter
	for(i=0;i<6;i++){
		filter_in[i].filter_priority = 0;
		filter_in[i].filter_type = 0;
		filter_in[i].filter_cmd = 0;
		filter_in[i].filter_dest = Can_Addr;
	}

	//DISABLE NESTED INTERRUPTS
	//INTCON1bits.NSTDIS = 1;

	//Reading configuration first
	ReadGlobalVariables(&g_globalVariables);
	

	//SET CONTROL MODE
	for (i = 0; i < 4; i++)
	{
		g_globalVariables.m_sharedVariables[i].CtrlMode = can_read_eeprom(CAN_EEPROM_STARTUP_MODE_ADDRESS);
			
		//TEST CONTROL MODE
		if(g_globalVariables.m_sharedVariables[i].CtrlMode >= CTRL_MODE_ERROR)
		{
			g_globalVariables.m_sharedVariables[i].CtrlMode = CTRL_MODE_CONFIG;
		}
	
		//Clear error code
		g_globalVariables.m_sharedVariables[i].ErrorCode = 0;
	}

	// init can
	can_init(filter_in,mask_in);
		
	//INITIALIZE ALL SENSORS
	sensors_init();

	//INIT PWM outputs and motor control	
	pwm_init();
		
	//This will block until we get our first ADC conversions...
	read_sensors();

	// Initialize TMR1 for 500 ms periodic ISR
	InitTMR1();	


	//SETUP CONTROL MODE
	//PID initialization
	for (i = 0; i< 4; i++)
	{
		pid_init(&g_globalVariables.m_sharedVariables[i]);
		safe_setpoint(&g_globalVariables.m_sharedVariables[i]);
	}

	//Initialize global variables
	g_globalVariables.m_ESTOPEnabled = (LATFbits.LATF6 == 0);
	g_globalVariables.m_WriteEEPROM = 0;
	g_globalVariables.m_loopTime = 0;

	// Initialize TMR4 for periodic ISR (PID LOOP)
	InitTMR4();	
	

	
	//START INFINITE LOOP
	while(1)
	{
		//READING CAN MESSAGES
		can_transceiver(Can_Addr);
		
		//HANDLE SAVE TO EEPROM HERE
		if (g_globalVariables.m_WriteEEPROM)
		{
			WriteGlobalVariables(&g_globalVariables);

			g_globalVariables.m_WriteEEPROM = 0;
		}


		//TODO SAFETY - GO BACK TO CONFIG MODE IF E-STOP IS ENABLED


	} //while(1)

	return 0;
}


/********************************************************************
void InitTMR1(void)	
*********************************************************************/

void InitTMR1(void)
{

	OpenTimer1(T1_ON & T1_GATE_OFF &
	T1_PS_1_256 & T1_SYNC_EXT_OFF &
	T1_SOURCE_INT, 3906); //10Hz	

	//Set priority = 2
	ConfigIntTimer1(T1_INT_PRIOR_2 & T1_INT_ON);
	return;
}


/************************************************************************
void InitTMR4(void)
*************************************************************************/

void InitTMR4(void)
{
	OpenTimer4(T4_ON & T4_IDLE_CON & T4_GATE_OFF
	& T4_PS_1_256  &
	T4_SOURCE_INT, 780); //100Hz	

	//Set priority = 4
	ConfigIntTimer4(T4_INT_PRIOR_4 & T4_INT_ON);

	return;
}

