//////////////////////////////////////////////////////////////////////
// sensors.c
//////////////////////////////////////////////////////////////////////
//
//	Description: 
//	Sensors reading and initialization
//
//	Author: Dominic Letourneau
//	Creation Date: 2006/11/06
//
//	Modification history:
//
//	Name					Date        Descrition
//	Dominic Letourneau		2009/04/28	Updated for 4x motor drive
//	Pierre Lepage			2008/05/14	Added read_sensor_input_capture_high() and _low()
//	Pierre Lepage			2008/05/13	Added Interrupt : IC4Interrupt()
//	Pierre Lepage			2008/05/13	Added global var : g_PulseWidthH,g_PulseWidthL,g_InputCap_1,g_InputCap_2,g_PusleCounter
//	Pierre Lepage			2008/05/13	Added init_input_capture() [Timer2 and IC4(RD11)]
//	Pierre Lepage			2008/05/13	Modification of sensor_read_input_Capture()
//	Pierre Lepage			2008/05/13	Modification of sensor_init()
//	Dominic Letourneau		2007/06/18	Changing the sign of the torque sensor
//	Dominic Letourneau		2007/04/27	Validation of Current measure (no change)
//	Dominic Letourneau		2007/04/19	New version of TTD
// 	Dominic Letourneau		2007/04/10  Position sensor code + using LATF,LATG instead of PORTF,PORTG
//  Dominic Letourneau		2007/03/29	Added external position sensor (SPI) in EXT2
//	Dominic	Letourneau		2007/03/27	Added external ADC (SPI) for torque sensor in EXT1
//	Dominic Letourneau		2006/11/22	Added sensor interface
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius / RoboMotio  2009
//////////////////////////////////////////////////////////////////////

#include "sensors.h"
#include "Device.h"
#include "utils.h"
#include "PWM.h"
#include <spi.h>
#include <timer.h>
#include <limits.h>

int g_inputCapHigh[4] = {0,0,0,0};
int g_inputCapLow[4] = {0,0,0,0};
long int g_adcValue[4] = {0,0,0,0};
long int g_adcAvgCount[4] = {0,0,0,0};

int g_currentValue[4] = {0,0,0,0};
int g_overTempValue[4] = {0,0,0,0};

unsigned char g_adcUpdated = 0;

//INPUT CAPTURE INTERRUPT
void __attribute__((__interrupt__(__save__(CORCON)),auto_psv)) _IC5Interrupt (void)
{
	//Clear the flag of interrupt
	IFS1bits.IC5IF = 0;
}

//INPUT CAPTURE INTERRUPT
void __attribute__((__interrupt__(__save__(CORCON)),auto_psv)) _IC6Interrupt (void)
{
	//Clear the flag of interrupt
	IFS1bits.IC6IF = 0;
}

//INPUT CAPTURE INTERRUPT
void __attribute__((__interrupt__(__save__(CORCON)),auto_psv)) _IC7Interrupt (void)
{
	//Clear the flag of interrupt
	IFS1bits.IC7IF = 0;

}



//INPUT CAPTURE INTERRUPT
void __attribute__((__interrupt__(__save__(CORCON)),auto_psv)) _IC8Interrupt (void)
{
	//Clear the flag of interrupt
	IFS1bits.IC8IF = 0;

/*	
	if(PORTDbits.RD11){
		//Rising edge...  Read all buffer...
		while (IC4CONbits.ICBNE)
		{
			g_InputCap_1 = IC4BUF; // reads the input capture buffer 
		}
		//Time off (Low)
		g_PulseWidthLow = g_InputCap_1;
		g_PulseCounter++;
	}
	else{
		//Falling edge
		while (IC4CONbits.ICBNE)
		{
			g_InputCap_2 = IC4BUF; // reads the input capture buffer 
		}

		if(g_PulseCounter){
			//Read g_InputCap
			if(g_InputCap_2 > g_InputCap_1){
				//Time on (High)
				g_PulseWidthHigh = g_InputCap_2 - g_InputCap_1;
			}		
		}
		g_InputCap_1 = 0;
		g_InputCap_2 = 0;
		g_PulseCounter = 0;
		TMR2 = 0;
	}
*/
	
}

//ADC INTERRUPT WILL OCCUR WHEN ALL ADC VALUES ARE TAKEN
void __attribute__((interrupt(__save__(CORCON)),auto_psv)) _ADCInterrupt (void)
{
	IFS0bits.ADIF = 0;
	
	g_adcValue[MOTOR_1] += ADCBUF3;
	g_adcValue[MOTOR_2] += ADCBUF2;
	g_adcValue[MOTOR_3] += ADCBUF1;
	g_adcValue[MOTOR_4] += ADCBUF0;

	g_adcAvgCount[MOTOR_1]++;
	g_adcAvgCount[MOTOR_2]++;
	g_adcAvgCount[MOTOR_3]++;
	g_adcAvgCount[MOTOR_4]++;

	//0.13 volts per amp
	g_currentValue[MOTOR_1] = ADCBUF7;
	g_currentValue[MOTOR_2] = ADCBUF6;
	g_currentValue[MOTOR_3] = ADCBUF5;
	g_currentValue[MOTOR_4] = ADCBUF4;

	//Convert to mA
	g_currentValue[MOTOR_1] *= 10;
	g_currentValue[MOTOR_2] *= 10;
	g_currentValue[MOTOR_3] *= 10;
	g_currentValue[MOTOR_4] *= 10;

	g_adcUpdated = 1;

	return;
}

//TODO SPI INTERRUPT HERE?

void init_analog()
{

/*
	The A/D module has six Control and Status registers. These registers are:
	 ADCON1: A/D Control Register 1
	 ADCON2: A/D Control Register 2
	 ADCON3: A/D Control Register 3
	 ADCHS: A/D Input Channel Select Register
	 ADPCFG: A/D Port Configuration Register
	 ADCSSL: A/D Input Scan Selection Register
	
	The ADCON1, ADCON2 and ADCON3 registers control the operation of the A/D module. The
	ADCHS register selects the input pins to be connected to the S/H amplifiers. The ADPCFG
	register configures the analog input pins as analog inputs or as digital I/O. The ADCSSL register
	selects inputs to be sequentially scanned.

	AD Result buffer :

	The module contains a 16-word dual port RAM, called ADCBUF, to buffer the A/D results. The
	16 buffer locations are referred to as ADCBUF0, ADCBUF1, ADCBUF2, ...., ADCBUFE,
	ADCBUFF.

*/

	//Disable AD Module
	ADCON1bits.ADON = 0;

	//Module configuration
	ADCON3bits.ADCS = 19; //Conversion clock (TAD) = (19 + 1) * 0.5 Tcy = 10 Tcy (minium is TAD = 333ns for good conversion) TAD = 500ns
	ADCON3bits.SAMC = 20; //AUTO SAMPLE TIME 20 TAD
	ADCON3bits.ADRC = 0; //0 = Clock derived from system clock
	//14TADS are required for conversion of one channel = 7us * 8 = 56us for 8 channels



	ADCON1bits.ADSIDL = 0; //Continue module operation in Idle mode
	ADCON1bits.FORM = 0; //Integer (DOUT = 0000 dddd dddd dddd)
	ADCON1bits.SSRC = 7; //Internal counter
	ADCON1bits.ASAM = 1; //Sampling begins immediately after last conversion completes. SAMP bit is auto set
	


	ADCON2bits.VCFG = 0; //000 = AVdd, AVss
	ADCON2bits.CSCNA = 1; //Scan inputs 
	ADCON2bits.SMPI = 7; //Interrupts at the completion of conversion of 8 samples
	ADCON2bits.BUFM = 0; //Buffer configured as one 16-word buffer ADCBUF(15...0.)
	ADCON2bits.ALTS = 0; //Always use MUX A input multiplexer settings
	

	//Set as input pins
	TRISBbits.TRISB0 = 1;
	TRISBbits.TRISB1 = 1;
	TRISBbits.TRISB2 = 1;
	TRISBbits.TRISB3 = 1;
	TRISBbits.TRISB8 = 1;
	TRISBbits.TRISB9 = 1;
	TRISBbits.TRISB10 = 1;
	TRISBbits.TRISB11 = 1;	

	//DEFAULT TO ALL DIGITAL INPUTS
	ADPCFG =0xFFFF;

	//Pots
	ADPCFGbits.PCFG3 = 0; //AN3 = ANALOG POT1
	ADPCFGbits.PCFG2 = 0; //AN2 = ANALOG POT2
	ADPCFGbits.PCFG1 = 0; //AN1 = ANALOG POT3
	ADPCFGbits.PCFG0 = 0; //AN0 = ANALOG POT4

	//Current sensors
	ADPCFGbits.PCFG11 = 0; //AN3 = ANALOG CS1
	ADPCFGbits.PCFG10 = 0; //AN2 = ANALOG CS2
	ADPCFGbits.PCFG9 = 0; //AN1 = ANALOG CS3
	ADPCFGbits.PCFG8 = 0; //AN0 = ANALOG CS4

	//Input scan channels
	ADCSSL = 0; //DEFAULT TO NO SCAN
	ADCSSLbits.CSSL0 = 1; //SELECT AN0 for input scan
	ADCSSLbits.CSSL1 = 1; //SELECT AN1 for input scan
	ADCSSLbits.CSSL2 = 1;//SELECT AN2 for input scan
	ADCSSLbits.CSSL3 = 1; //SELECT AN3 for input scan
	ADCSSLbits.CSSL8 = 1; //SELECT AN8 for input scan
	ADCSSLbits.CSSL9 = 1; //SELECT AN9 for input scan
	ADCSSLbits.CSSL10 = 1;//SELECT AN10 for input scan
	ADCSSLbits.CSSL11 = 1; //SELECT AN11 for input scan

	//Set interrupt priority = 5
	IPC2bits.ADIP = 5;
	
	IFS0bits.ADIF = 0; // clear ADC interrupt flag
	IEC0bits.ADIE = 1; // Enable interrupts

	//Enable AD Module
	ADCON1bits.ADON = 1;

	//Wait until we have made our first conversion
	while(g_adcUpdated == 0);
}

void init_input_capture()
{

	TRISDbits.TRISD4 = 1;	// RD4 Port in input, IC5
	TRISDbits.TRISD5 = 1;   // RD5 Port in input, IC6
	TRISBbits.TRISB4 = 1;   // RB4 Port in input, IC7
	TRISBbits.TRISB5 = 1;   // RB5 Port in input, IC8 	




	//TIMER 2 Configuration
	T2CONbits.TON = 0;		//Stop the timer 2.
	TMR2 =0; 				//Reset the value of Timer 2
	T2CONbits.TSIDL = 0;	//Continue module operation in Idle mode
	T2CONbits.TGATE = 0;	//No GATE accumulation
	
	T2CONbits.TCKPS = 1;	//Prescaler 1:8 (50 ns x 8 = 400 ns)[400 ns x 65535 = 26214 us buff]
							//-------- USING MAE3-P12 Sensor from USDigital ------------------
							//When 4096 => g_PulseWidth = 10240
							//When 1	=> g_PulseWidth = 2 or 3

	T2CONbits.T32 = 0;		//Used this timer as a 16bits
	T2CONbits.TCS = 0;		//Use internal clock
	
	//INPUT CAPTURE (IC5) Config
	IC5CONbits.ICSIDL = 0;	//Input capture module will continue to operate in CPU Idle mode
	IC5CONbits.ICTMR = 1;	//TMR2 contents are captured on capture event
	IC5CONbits.ICI = 0;		//Interrupt on every capture event
	IC5CONbits.ICM = 1;		//Capture mode, every edge
	IFS1bits.IC5IF = 0;		//Clear the flag
	IPC7bits.IC5IP = 3;		//Priority 
	IEC1bits.IC5IE = 1;		//Activate Interrupt


	//INPUT CAPTURE (IC6) Config
	IC6CONbits.ICSIDL = 0;	//Input capture module will continue to operate in CPU Idle mode
	IC6CONbits.ICTMR = 1;	//TMR2 contents are captured on capture event
	IC6CONbits.ICI = 0;		//Interrupt on every capture event
	IC6CONbits.ICM = 1;		//Capture mode, every edge
	IFS1bits.IC6IF = 0;		//Clear the flag
	IPC7bits.IC6IP = 3;		//Priority 
	IEC1bits.IC6IE = 1;		//Activate Interrupt

	//INPUT CAPTURE (IC7) Config
	IC7CONbits.ICSIDL = 0;	//Input capture module will continue to operate in CPU Idle mode
	IC7CONbits.ICTMR = 1;	//TMR2 contents are captured on capture event
	IC7CONbits.ICI = 0;		//Interrupt on every capture event
	IC7CONbits.ICM = 1;		//Capture mode, every edge
	IFS1bits.IC7IF = 0;		//Clear the flag
	IPC4bits.IC7IP = 3;		//Priority 
	IEC1bits.IC7IE = 1;		//Activate Interrupt

	//INPUT CAPTURE (IC8) Config
	IC8CONbits.ICSIDL = 0;	//Input capture module will continue to operate in CPU Idle mode
	IC8CONbits.ICTMR = 1;	//TMR2 contents are captured on capture event
	IC8CONbits.ICI = 0;		//Interrupt on every capture event
	IC8CONbits.ICM = 1;		//Capture mode, every edge
	IFS1bits.IC8IF = 0;		//Clear the flag
	IPC4bits.IC8IP = 3;		//Priority 
	IEC1bits.IC8IE = 1;		//Activate Interrupt	


	//START TIMER 2
	T2CONbits.TON = 1;		//Start the timer 2.	
}

void init_thermal_state(void)
{
	TRISGbits.TRISG9 = 1; //Thermal 1 input
	TRISGbits.TRISG7 = 1; //Thermal 2 input
	TRISGbits.TRISG6 = 1; //Thermal 3 input
	TRISGbits.TRISG8 = 1; //Thermal 4 input
}

void sensors_init(void)
{
	//INIT ANALOG
	init_analog();

	//INIT Input Capture
	init_input_capture();

	//INIT Thermal state
	init_thermal_state();
}



signed int sensors_read_input_capture(MotorId id)
{
	return 0;
}

signed int sensors_read_adc(MotorId id)
{
	int myValue;
	

	//PROTECT LONG VARIABLES
	__asm__ volatile ("disi #0x3FFF");
	if (g_adcAvgCount[id] > 0)
	{
		//Calculate average
		myValue = g_adcValue[id] / g_adcAvgCount[id];
	}
	else
	{
		//return error
		myValue = -1;
	}

	g_adcValue[id] = 0;
	g_adcAvgCount[id] = 0;
	__asm__ volatile ("disi #0x0000");

	return myValue;
}


signed int sensors_read_current(MotorId id)
{
	return g_currentValue[id];
}


signed int sensors_read_thermal_state(MotorId id)
{
	switch(id)
	{
		case MOTOR_1:
		if (PORTGbits.RG9)
		{
			//INCREMENT COUNTER (AVOID OVERFLOW)
			if (g_overTempValue[MOTOR_1] < INT_MAX)
				g_overTempValue[MOTOR_1]++;
		}
		else
		{
			//RESET FLAG
			g_overTempValue[MOTOR_1] = 0;
		}

		break;
	
		case MOTOR_2:

		if (PORTGbits.RG7)
		{
			//INCREMENT COUNTER (AVOID OVERFLOW)
			if (g_overTempValue[MOTOR_2] < INT_MAX)
				g_overTempValue[MOTOR_2]++;
		}
		else
		{
			//RESET FLAG
			g_overTempValue[MOTOR_2] = 0;
		}

		break;

		case MOTOR_3:

		if (PORTGbits.RG6)
		{
			//INCREMENT COUNTER (AVOID OVERFLOW)
			if (g_overTempValue[MOTOR_3] < INT_MAX)
				g_overTempValue[MOTOR_3]++;
		}
		else
		{
			//RESET FLAG
			g_overTempValue[MOTOR_3] = 0;
		}
		break;

		case MOTOR_4:

		if (PORTGbits.RG8)
		{
			//INCREMENT COUNTER (AVOID OVERFLOW)
			if (g_overTempValue[MOTOR_4] < INT_MAX)
				g_overTempValue[MOTOR_4]++;
		}
		else
		{
			//RESET FLAG
			g_overTempValue[MOTOR_4] = 0;
		}
		break;

	}

	return g_overTempValue[id];
}


