//////////////////////////////////////////////////////////////////////
//  SwitchingModes.c
//////////////////////////////////////////////////////////////////////
//
//  Description: 
//  Multiple switching modes available for the motor.
//
//  Author: Dominic Letourneau
//  Creation Date: 2006/11/06
//
//  Modification history:
//
//  Name					Date        Descrition
//	Dominic Letourneau		2007/10/25	Added IDLE mode for motors when in config mode
//	Dominic Letourneau		2007/09/04	Disable PWM Ramping. Useless...
//	Dominic Letourneau		2007/06/04	PWM Ramping, ADC TRIGERRING MOVED DIRECTLY IN THE SWITCHING FUNCTIONS
//	Dominic Letourneau		2007/05/16	Make sure bootstraps capacitor are charged at startup
//	Dominic	Letourneau		2007/05/09	Added capture of FAULT,UVFLT, OVFLT from A3935
//	Dominic Letourneau		2007/05/03	Avoiding interrupts while changing duty cycle. This caused glitches in switching.
//	Dominic Letourneau		2007/05/03	Fixed Fault input to avoid PWM output when a fault occurs
//  Dominic Letourneau		2007/04/27	Calling getMaxDuty() and getMinDuty() to have a single place for calculation of min,max
//  Dominic Letourneau		2007/02/08  Fixed Maximum duty cycle for edge aligned modes and center aligned modes
//										Resolution is Tcy  / 2 for PDCx for both modes
//	Dominic Letourneau		2007/02/08	Validation of init mode, default to BLDC_SR_SD if invalid
//  Dominic Letourneau		2007/01/19	Added correct configuration for ADC trigers
//	Dominic Letourneau		2006/11/30	Added brushed modes
//	Dominic Letourneau		2006/11/07  Validation of all modes, fixing sinus like direction					
//	Dominic Letourneau		2006/11/06	Added brushless modes
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius / RoboMotio  2006-2007
//////////////////////////////////////////////////////////////////////
#include "SwitchingModes.h"
#include "Device.h"
#include <timer.h>
#include "utils.h"


//declare a typedef for a switching function pointer
typedef void (*BLDCSwitchingFunctionPtr) (signed char, signed int);

//switching function prototypes
//BLDC motors
void BLDC_NSR_FD_Switching(signed char _Sector, signed int _Duty);
void BLDC_SR_SD_Switching(signed char _Sector, signed int _Duty);
void BLDC_NSR_SD_Switching(signed char _Sector, signed int _Duty);
void BLDC_SR_SD_SINUS_Switching(signed char _Sector, signed int _Duty);

//The current switching function pointer
//We are using a function pointer to avoid testing
//the mode every time we have an interrupt for hall sensors.
//It is also useful when updating the current duty cycle.
BLDCSwitchingFunctionPtr g_BLDCSwitchingFunction = NULL;


//The actual switching mode
unsigned char g_switchingMode = 0;

//The current duty cycle
signed int g_currentDuty = 0;

//IDLE Enabled flag
unsigned char g_idleEnabled = 0;

//The current motor direction
unsigned char g_currentDirection = CW;

//The hall switching counter, can be used to 
//calculate motor speed
signed int g_hallSwitchingCounter = 0;

//Lookup table to select the switching transistors according
//to the hall sensors value
// 0 (no hall) or (7) all halls are invalid values ---> -1
//Hall sensors are disposed in a way to have only one bit change
//at the time (gray code) when rotating the rotor (CCW,CW)
const char g_sectorTable[] = {-1,4,2,3,0,5,1,-1}; 


//Actual sector, must be initialized properly 
//by reading hall sensors value for BLDC mode
unsigned char g_currentSector = 0;
unsigned char g_lastSector = -1; 


//PWM RAMPING
void InitTMR4(void);
void pwm_ramp_handle();
signed int g_desiredDuty = 0;



//This table must be intialized according to selected switching mode
//by default, no switching will occur, forcing PWM pin to "0" or GND.
unsigned int g_BLDC_StateLoTable[] = {0x0000,0x0000,0x0000,0x0000,0x0000,0x0000};

//idle mode == no motor braking
unsigned int g_BLDC_StateLoTable_IDLE[] = {0x0000,0x0000,0x0000,0x0000,0x0000,0x0000};

//Mode = Non Sync. rectification, fast decay
//*WARNING* Complementary mode must be disabled for this mode to work.
//Send PWM in Hi and Lo transistors, current goes through diodes at PWM inactive state.
const unsigned int BLDC_StateLoTable_NSR_FD[] = {	0x0600,  /* PWM2L -> PWM, PWM1H -> PWM */
							 	0x1200,  /* PWM3L -> PWM, PWM1H -> PWM */
                                0x1800,  /* PWM3L -> PWM, PWM2H -> PWM */
								0x0900,  /* PWM1L -> PWM, PWM2H -> PWM */
								0x2100,  /* PWM1L -> PWM, PWM3H -> PWM */
								0x2400}; /* PWM2L -> PWM, PWM3H -> PWM */


//Mode = Sync. rectification, slow decay
//*WARNING* Complementary mode must be enabled for the dead time to be inserted
//dead time is variable (according to transistors specs, normally around 100-500 ns).
//"Override" bit is enabled for Hi-Lo pair containing the Hi part that is required
//Other Lo part must always be "ON". Current goes through transistors at PWM inactive 
//state to avoid heat loss.
const unsigned int BLDC_StateLoTable_SR_SD[] = { 0x0304,  /* PWM2L -> 1, PWM1H,PWM1L -> PWM */
							 	      0x0310,  /* PWM3L -> 1, PWM1H,PWM1L -> PWM */
                                      0x0C10,  /* PWM3L -> 1, PWM2H,PWM2L -> PWM */
								      0x0C01,  /* PWM1L -> 1, PWM2H,PWM2L -> PWM */
								      0x3001,  /* PWM1L -> 1, PWM3H,PWM3L -> PWM */
								      0x3004}; /* PWM2L -> 1, PWM3H,PWM3L -> PWM */

//Mode = Non Sync. rectification, slow decay
//*WARNING* Complementary mode can be "on" or "off"
//Switching the Hi part with the PWM, Lo part stays "ON". 
//Current goes through transistors at PWM inactive state.
const unsigned int BLDC_StateLoTable_NSR_SD[] = { 0x0204,  /* PWM2L -> 1, PWM1H -> PWM */
							 	       0x0210,  /* PWM3L -> 1, PWM1H -> PWM */
                                       0x0810,  /* PWM3L -> 1, PWM2H -> PWM */
								       0x0801,  /* PWM1L -> 1, PWM2H -> PWM */
								       0x2001,  /* PWM1L -> 1, PWM3H -> PWM */
								       0x2004}; /* PWM2L -> 1, PWM3H -> PWM */

//CENTER ALIGNED-COMPLEMENTARY
//Mode = Sync. rect, slow decay, sinus like
//*WARNING* Complementary mode must be "ON" for this mode to work.
//This mode will switch Both transistor pairs with different PWM values
//Conduction in the motor will occur at two different time in the cycle generating
//a sinus-like waveform in the motor's current. At PWM inactive state, the current will
//flow in the Hi part of transistor then in the Lo part, avoiding heat dissipation in the
//diodes.
const unsigned int BLDC_StateLoTable_SR_SD_SINUS[] = {	0x0F00,  // PWM2H, PWM2L -> PWM, PWM1H,PWM1L -> PWM
							 	0x3300,  // PWM3H, PWM3L -> PWM, PWM1H,PWM1L -> PWM 
                                0x3C00,  // PWM3H, PWM3L -> PWM, PWM2H,PWM2L -> PWM 
								0x0F00,  // PWM1H, PWM1L -> PWM, PWM2H,PWM2L -> PWM 
								0x3300,  // PWM1H, PWM1L -> PWM, PWM3H,PWM3L -> PWM 
								0x3C00}; // PWM2H, PWM2L -> PWM, PWM3H,PWM3L -> PWM 

/*********************************************************************
Interrupt routines for hall sensors (__save__())

WREG8 AND WREG10 MUST BE SAVED, SINCE THEY ARE USED IN THE PID CALCULATION
NESTED INTERRUPTS ARE ALLOWED, SO T3INTERRUPT (PID) CAN BE INTERRUPTED BY
THIS HIGHER INTERRUPT FUNCTION.
**********************************************************************/
//void __attribute__((__interrupt__(__save__(CORCON)))) _CNInterrupt (void)
void __attribute__((interrupt(__save__(CORCON)),auto_psv)) _CNInterrupt (void)
{		
	int i = 0;
	unsigned int temp = ((PORTD >> 4) & 0x0007);
	unsigned int temp2 = ((PORTD >> 4) & 0x0007);

	while(1) {
		if(temp == temp2)
			i++;
		else 
			i = 0;
		temp = temp2;
		temp2 = ((PORTD >> 4) & 0x0007);
		if(i > 5) 
			break;
	}
		

	g_currentSector = g_sectorTable[temp2];	// Get Sector from table

	if (g_currentSector != g_lastSector)	// This is a MUST for getting around the HW slow rate
	{

		//Looking for which direction we are going
		if (g_currentSector == (g_lastSector + 1)
			|| ((g_currentSector == 0) && (g_lastSector == 5)) )
		{
			g_currentDirection = CCW;
			g_hallSwitchingCounter--;
		}
		else
		{
			g_currentDirection = CW;
			g_hallSwitchingCounter++;
		}
	}

	if (g_idleEnabled)
	{
		OVDCON = 0;
		PDC1 = PDC2 = PDC3 = 0;	
	}
	else
	{
		//If g_currentDuty > 0 will run CW, else will run CCW
		g_BLDCSwitchingFunction(g_currentSector, g_currentDuty); 
	}

	// If a change in sector, update last sector
	g_lastSector = g_currentSector;	

	IFS0bits.CNIF = 0;	// Clear interrupt flag

	return;
}

/*********************************************************************
	In the topology used, it is necessary to charge the bootstrap
	caps each time the motor is energized for the first time after an
	undetermined amount of time. ChargeBootstraps subroutine turns ON
	the lower transistors for 10 ms to ensure voltage on these caps,
	and then it transfers the control of the outputs to the PWM
	module.
*********************************************************************/
void InitializeBootstraps(void)
{
	unsigned int i;
	OVDCON = 0x0015;	// Turn ON low side transistors to charge
	for (i = 0; i < 33330; i++) // 10 ms Delay at 20 MIPs
		;
	OVDCON = 0x0000;
	return;
}

/*********************************************************************
void InitSwitchingMode(SwitchingMode mode, unsigned int frequency)
**********************************************************************/
void InitSwitchingMode(SwitchingMode mode, unsigned int frequency)
{
	unsigned char i = 0;
	const unsigned int* StateLoTablePtr = NULL;

	//Charge bootstraps capacitors
	InitializeBootstraps();

	//MAKE SURE NO INTERRUPTS HAPPEN IN THIS FUNCTION
	__asm__ volatile ("disi #0x3FFF");

	//VALID MODE TEST
	//IF NOT VALID, USING BLDC_SR_SD
	if (mode > BDC_SR_SD_SINUS)
	{
		mode = BLDC_SR_SD;
	}

	//save current switching mode
	g_switchingMode = mode;

	//set current duty
	g_currentDuty = 0;

	//Initialize MCPWM module
	TRISE = 0x0300;		// PWM pins as outputs
	TRISD |= 0xFFF0;    //ALL D Port input

	OVDCON = 0x0000;	// Disable all PWM outputs.
	
	PDC1 = 0;			// Duty cycle = 0
	PDC2 = 0;			// Duty cycle = 0
	PDC3 = 0;			// Duty cycle = 0
	
	//TODO : DEAD TIME SHOULD BE A PARAMETER?
	//DTCON1 = 0x0505;	// 250ns dead time
	//DTCON1 = 0x0A0A;	// 500ns dead time
	//DTCON1 = 0x1414;	// 1000ns dead time
	DTCON1 = 0x1F1F;	// 1600ns dead time

	FLTACON = 0b0011111110000111; //FAULTA ENABLED
								  //PAIR 1,2 & 3
								  //FAULT STATE = PWM = 0 (OFF)
								  //MODE = CYCLE BY CYCLE
	

	switch(mode)
	{
		case BLDC_NSR_FD: 
		case BDC_NSR_FD: //Brushed and Brushless DC motor, non sync. fast decay mode
			// Compute Period based on CPU speed 
			//and required PWM frequency (see defines)
			PTPER = (FREQ_CYCLE/PWM_FREQUENCY - 1);	
			
			//Enable PWM output pins and configure them as 
			//non-complementary mode		 
			PWMCON1 = 0x0777;	
			
			// start PWM as edge aligned mode
			PTCON = 0x8000;	

			StateLoTablePtr = &BLDC_StateLoTable_NSR_FD[0];

			//Initialize switching function pointer
			g_BLDCSwitchingFunction = BLDC_NSR_FD_Switching;
		break;

		case BLDC_SR_SD: 
		case BDC_SR_SD://Brushed and Brushless DC motor, sync. rect., slow decay
			// Compute Period based on CPU speed 
			//and required PWM frequency (see defines)
			PTPER = (FREQ_CYCLE/PWM_FREQUENCY - 1);	

			//Enable PWM output pins and configure them as 
			//complementary mode		 
			PWMCON1 = 0x0077;

			// start PWM as edge aligned mode, 1:1 prescaler, 1:1 postscaler
			PTCON = 0x8000;	

			StateLoTablePtr = &BLDC_StateLoTable_SR_SD[0];

			//Initialize switching function pointer
			g_BLDCSwitchingFunction = BLDC_SR_SD_Switching;
		break;

		case BLDC_NSR_SD: 
		case BDC_NSR_SD://Brushed and Brushless DC motor, non sync., slow decay
			// Compute Period based on CPU speed 
			//and required PWM frequency (see defines)
			PTPER = (FREQ_CYCLE/PWM_FREQUENCY - 1);	
			
			//Enable PWM output pins and configure them as 
			//complementary mode		 
			PWMCON1 = 0x0077;

			// start PWM as edge aligned mode
			PTCON = 0x8000;	

			StateLoTablePtr = &BLDC_StateLoTable_NSR_SD[0];

			//Initialize switching function pointer
			g_BLDCSwitchingFunction = BLDC_NSR_SD_Switching;
		break;

		case BLDC_SR_SD_SINUS: 
		case BDC_SR_SD_SINUS://Brushed and Brushless DC motor, sync. slow decay, Sinus-like
			// Compute Period based on CPU speed 
			//and required PWM frequency (see defines)
			PTPER = (FREQ_CYCLE/PWM_FREQUENCY - 1) / 2;	

			//Enable PWM output pins and configure them as 
			//complementary mode		 
			PWMCON1 = 0x0077;

			// start PWM as center aligned mode
			//single interrupt, 1:1 prescaler, 1:1 postscaler
			PTCON = 0x8002;	

			StateLoTablePtr = &BLDC_StateLoTable_SR_SD_SINUS[0];

			//Initialize switching function pointer
			g_BLDCSwitchingFunction = BLDC_SR_SD_SINUS_Switching;
		break;
	}

	//ADC TRIGERRING
	SEVTCMPbits.SEVTDIR = 0;//Special event trigger will occur when the PWM time base is counting upwards.
	

	//HALL SENSORS FOR BLDC MODES ONLY
	if (mode <= BLDC_SR_SD_SINUS && g_BLDCSwitchingFunction != NULL && StateLoTablePtr != NULL)
	{
		//copy correct lookup table
		//for the selected switching mode
		for (i = 0; i < 6; i++)
		{
			g_BLDC_StateLoTable[i] = StateLoTablePtr[i];
		}

		//Initialize Hall pins
		//Hall A -> CN13  Hall A is used for commutation.
		//Hall B -> CN14  Hall B is used for commutation.
		//Hall C -> CN15. Hall C is used for commutation.
	
		// Init Input change notification 
    	CNEN1bits.CN13IE = 1;	// Enable CN13
		CNEN1bits.CN14IE = 1;	// Enable CN14
		CNEN1bits.CN15IE = 1;	// Enable CN15

		//Set CN Priority to highest priority (6)
		IPC3bits.CNIP = 6;

		// Read halls and initialize Sector variable
		g_lastSector = g_currentSector = g_sectorTable[(unsigned int)((PORTD >> 4) & 0x0007)];

		// Clear all interrupts flags
		IFS0bits.CNIF = 0;	// Clear interrupt flag

		// enable CN interrupts
		__asm__ volatile("DISI #1");
		IEC0bits.CNIE = 1;	// Enable interrupts on CN pins

		//send 0 duty cycle
		setPWMDuty(0,0);
	}
	else 
	{
		//PULL UP ON HALL EFFECT INPUTS
		//CN13, CN14, CN15
		CNPU1 |= 0x0007;

		//copy correct lookup table
		//for the selected switching mode
		if (StateLoTablePtr != NULL)
		{
			for (i = 0; i < 6; i++)
			{
				g_BLDC_StateLoTable[i] = StateLoTablePtr[i];
			}
		}

		//send 0 duty cycle
		setPWMDuty(0,0);
		
	}
	__asm__ volatile ("disi #0x0000");
}

/*********************************************************************
void setPWMDuty(signed int duty)
**********************************************************************/
void setPWMDuty(signed int duty, unsigned char idle_enabled)
{
	//LIMIT DUTY VALUE
	signed int min_duty = getMinDuty();
	signed int max_duty = getMaxDuty();


	duty = MAX(min_duty,MIN(duty,max_duty)); 		

	//save current duty
	g_currentDuty = duty;
	
	//store idle state
	g_idleEnabled = idle_enabled;

	//Call appropriate switching function if idle is disabled
	if (g_idleEnabled)
	{
		//force duty cycle to zero
		g_currentDuty = 0;

		//configure 
		__asm__ volatile ("disi #0x3FFF");
		OVDCON = 0;
		PDC1 = PDC2 = PDC3 = 0;
		__asm__ volatile ("disi #0x0000");
	}
	else if(g_BLDCSwitchingFunction != NULL)
	{
		
		//ADC TRIGERRING MOVED DIRECTLY IN THE SWITCHING FUNCTIONS

		//Call the BLDC switching mechanism with equivalent sector = 0 (default value in BDC motors)
		//to use only H-Bridge created with PWM1H,PWM1L, PWM2H,PWM2L
		//Negative duty will be handled in this function


		//THIS FUNCTION MUST NOT BE CALLED WHEN AN INTERRUPT COULD OCCUR
		//THIS WOULD RESULT IN SWITCHING GLITCHES
		__asm__ volatile ("disi #0x3FFF");
		g_BLDCSwitchingFunction(g_currentSector,duty);
		__asm__ volatile ("disi #0x0000");

	}
}

/*********************************************************************
signed int getPWMDuty(void)
**********************************************************************/
signed int getPWMDuty(void) 
{
	return g_currentDuty;
}

/*********************************************************************
unsigned char getDirection(void)
**********************************************************************/
unsigned char getDirection(void) 
{
	return g_currentDirection;
}

/*********************************************************************
signed int getHallSwitchingCounter() 
**********************************************************************/
signed int getHallSwitchingCounter() 
{
	return g_hallSwitchingCounter;
}

/*********************************************************************
void resetHallSwitchingCounter()
**********************************************************************/
void resetHallSwitchingCounter()
{
	g_hallSwitchingCounter = 0;
}

/*********************************************************************
void BLDC_NSR_FD_Switching(signed char _Sector, signed int _Duty)
**********************************************************************/
void BLDC_NSR_FD_Switching(signed char _Sector, signed int _Duty)
{
	//duty cycle range [-PTPER + 1, PTPER - 1]
	//already tested in higher level function

	//Sector out of range protection
	if (_Sector == -1)
		OVDCON = 0x0000; // Invalid sector will shut down all transistors
	else {
		//CW
		if (_Duty >= 0)
		{
			//Current_Direction = CW;
			OVDCON = g_BLDC_StateLoTable[_Sector];
		}
		//CCW
		else
		{
			//Current_Direction = CCW;
			OVDCON = g_BLDC_StateLoTable[(_Sector + 3) % 6];
			_Duty = -_Duty;
		}

		//SETUP ADC TRIGGERING
		SEVTCMP = (_Duty >> 1);
		PWMCON2 = 0x0000;		// 1:1 postscale values, for special event trigger, must be written every time AFTER
								// we change the value of SEVTCMP

		//APPLY DUTY CYCLE
		PDC1 = PDC2 = PDC3 = _Duty;

	}
	return;
}

/*********************************************************************
void BLDC_SR_SD_Switching(signed char _Sector, signed int _Duty)
**********************************************************************/
void BLDC_SR_SD_Switching(signed char _Sector, signed int _Duty)
{
	//duty cycle range [-PTPER + 1, PTPER - 1]
	//already tested in higher level function

	//Sector out of range protection
	if (_Sector == -1)
		OVDCON = 0x0000; // Invalid sector will shut down all transistors
	else {
		//CW
		if (_Duty >= 0)
		{
			//Current_Direction = CW;
			OVDCON = g_BLDC_StateLoTable[_Sector];
		}
		//CCW
		else
		{
			//Current_Direction = CCW;
			OVDCON = g_BLDC_StateLoTable[(_Sector + 3) % 6];
			_Duty = -_Duty;
		}

		//SETUP ADC TRIGGERING
		SEVTCMP = (_Duty >> 1);
		PWMCON2 = 0x0000;		// 1:1 postscale values, for special event trigger, must be written every time AFTER
								// we change the value of SEVTCMP

		//APPLY DUTY CYCLE
		PDC1 = PDC2 = PDC3 = _Duty;

	}
	return;
}

/*********************************************************************
void BLDC_NSR_SD_Switching(signed char _Sector, signed int _Duty)
**********************************************************************/
void BLDC_NSR_SD_Switching(signed char _Sector, signed int _Duty)
{
	//duty cycle range [-PTPER + 1, PTPER - 1]
	//already tested in higher level function

	//Sector out of range protection
	if (_Sector == -1)
		OVDCON = 0x0000; // Invalid sector will shut down all transistors
	else {
		//CW
		if (_Duty >= 0)
		{
			//Current_Direction = CW;
			OVDCON = g_BLDC_StateLoTable[_Sector];
		}
		//CCW
		else
		{
			//Current_Direction = CCW;
			OVDCON = g_BLDC_StateLoTable[(_Sector + 3) % 6];
			_Duty = -_Duty;
		
		}
		//SETUP ADC TRIGGERING
		SEVTCMP = (_Duty >> 1);
		PWMCON2 = 0x0000;		// 1:1 postscale values, for special event trigger, must be written every time AFTER
								// we change the value of SEVTCMP

		//APPLY DUTY CYCLE
		PDC1 = PDC2 = PDC3 = _Duty;

	}
	return;
}

/*********************************************************************
void BLDC_SR_SD_SINUS_Switching(signed char _Sector, signed int _Duty)
**********************************************************************/
void BLDC_SR_SD_SINUS_Switching(signed char _Sector, signed int _Duty)
{
	//duty cycle range [-PTPER + 1, PTPER - 1]
	//already tested in higher level function

	//Sector out of range protection
	if (_Sector == -1)
		OVDCON = 0x0000; // Invalid sector will shut down all transistors
	else {
		//CW
		if (_Duty >= 0)
		{
			//Current_Direction = CW;
			OVDCON = g_BLDC_StateLoTable[_Sector];
		}
		//CCW
		else
		{
			//Current_Direction = CCW;
			OVDCON = g_BLDC_StateLoTable[(_Sector + 3) % 6];
		}

		//APPLY PROPER DUTY CYCLE
		//WE ARE LUCKY THAT SIGNS DO FIT EVEN IN CCW MODE
		//WITHOUT UPDATING THE _Sector VARIABLE and _Duty SIGN (+/-)
		switch(_Sector)
		{
			case 0: // PWM2H, PWM2L -> PWM, PWM1H,PWM1L -> PWM
			PDC1 = PTPER + _Duty;
			PDC2 = PTPER - _Duty;
			PDC3 = 0;
			break;

			case 1: // PWM3H, PWM3L -> PWM, PWM1H,PWM1L -> PWM
			PDC1 = PTPER + _Duty;
			PDC2 = 0;
			PDC3 = PTPER - _Duty;
			break;

			case 2: // PWM3H, PWM3L -> PWM, PWM2H,PWM2L -> PWM 
			PDC1 = 0;
			PDC2 = PTPER  + _Duty;
			PDC3 = PTPER  - _Duty;
			break;

			case 3: // PWM1H, PWM1L -> PWM, PWM2H,PWM2L -> PWM 
			PDC1 = PTPER - _Duty;
			PDC2 = PTPER + _Duty;
			PDC3 = 0;
			break;

			case 4: // PWM1H, PWM1L -> PWM, PWM3H,PWM3L -> PWM
			PDC1 = PTPER - _Duty;
			PDC2 = 0;
			PDC3 = PTPER + _Duty;
			break;

			case 5:// PWM2H, PWM2L -> PWM, PWM3H,PWM3L -> PWM
			PDC1 = 0;
			PDC2 = PTPER - _Duty;
			PDC3 = PTPER + _Duty;
			break;
		} //switch (_Sector)

		//SETUP ADC TRIGGERING
		//SINUS LIKE MODE WILL SAMPLE
		//AT PTPER, MIDDLE OF THE ACTIVE STATE OF FIRST SINUS PHASE
		SEVTCMP = PTPER;	    // Enable triggering for ADC in the middle of duty cycle active state
		PWMCON2 = 0x0000;		// 1:1 postscale values, for special event trigger, must be written every time AFTER
								// we change the value of SEVTCMP

	} //else (_Sector == -1)
	return;
}


signed int getMaxDuty() {

	signed int PTPERVal = PTPER;

	//DO NOT ALLOW 100% DUTY
	if (g_switchingMode != BLDC_SR_SD_SINUS && g_switchingMode != BDC_SR_SD_SINUS)
	{
		return 2*PTPERVal -1;
	}
	else
	{
		return PTPERVal -1;
	}
}

signed int getMinDuty()
{
	return -1 * getMaxDuty();
}


unsigned char getSwitchingStatus()
{
	unsigned char status = 0;
	status = PORTD & 0x0007;
	return status;
}

