; WARRANTY OF ANY KIND, WHETHER EXPRESS OR IMPLIED, INCLUDING BUT NOT
; LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
; PARTICULAR PURPOSE, OR NON-INFRINGEMENT. IN NO EVENT SHALL MICROCHIP
; BE LIABLE FOR ANY INCIDENTAL, SPECIAL, INDIRECT OR CONSEQUENTIAL
; DAMAGES, LOST PROFITS OR LOST DATA, HARM TO YOUR EQUIPMENT, COST OF
; PROCUREMENT OF SUBSTITUTE GOODS, TECHNOLOGY OR SERVICES, ANY CLAIMS
; BY THIRD PARTIES (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF),
; ANY CLAIMS FOR INDEMNITY OR CONTRIBUTION, OR OTHER SIMILAR COSTS.
;
; REVISION HISTORY:
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
; Author            Date      Comments on this revision
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
; First Last Name   mm/dd/yy  Place your comments here
; Jorge Zambada     08/17/05  Assembly functions for WIB
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

.include "p30f5011.inc"

	.global _SpeedCalculation
	.global _SpeedControl

;********************************************************************
; This subroutine Calculates the speed of the BLDC motor in fractional
; format using Fractional Division as follows:
;
;                                        Minimum Period
;   Actual Speed = (Fractinal Division) -----------------
;                                        Actual Period
;
;********************************************************************
;//SIGNED FRAC DIVIDE
_SpeedCalculation:
	MOV __MINPERIOD, W8
	MOV _Period, W9
	REPEAT #17
	DIVF W9, W8
	MOV W0, _Speed
	RETURN

;********************************************************************
;
;                                             ----   Proportional
;                                            |    |  Output
;                             ---------------| Kp |-----------------
;                            |               |    |                 |
;                            |                ----                  |
;Reference                   |                                     --- 
;Speed         ---           |           --------------  Integral | + | Control   -------
;     --------| + |  Error   |          |      Ki      | Output   |   | Output   |       |
;             |   |----------|----------| ------------ |----------|+  |----------| Plant |--
;        -----| - |          |          |  1 - Z^(-1)  |          |   |          |       |  |
;       |      ---           |           --------------           | + |           -------   |
;       |                    |                                     ---                      |
;       | Measured           |         -------------------  Deriv   |                       |
;       | Speed              |        |                   | Output  |                       |
;       |                     --------| Kd * (1 - Z^(-1)) |---------                        |
;       |                             |                   |                                 |
;       |                              -------------------                                  |
;       |                                                                                   |
;       |                                                                                   |
;        -----------------------------------------------------------------------------------
;
;   ControlOutput(K) = ControlOutput(K-1) 
;                    + ControlDifference(K) * (Kp + Ki + Kd)
;                    + ControlDifference(K-1) * (-Kp - 2*Kd)
;                    + ControlDifference(K-2) * Kd
;
;   Using PIDCoefficients:
;   PIDCoefficients[0] = Kp + Ki + Kd
;   PIDCoefficients[1] = -(Kp + 2*Kd)
;   PIDCoefficients[2] = Kd
;   and leting:
;   ControlOutput -> ControlOutput(K) and ControlOutput(K-1)
;   ControlDifference[0] -> ControlDifference(K)
;   ControlDifference[1] -> ControlDifference(K-1)
;   ControlDifference[2] -> ControlDifference(K-2)
;
;   ControlOutput = ControlOutput
;                 + ControlDifference[0] * PIDCoefficients[0]
;                 + ControlDifference[1] * PIDCoefficients[1]
;                 + ControlDifference[2] * PIDCoefficients[2]
;
;   This was implemented using Assembly with signed fractional and saturation enabled
;   with MAC instruction
;********************************************************************

_SpeedControl:
	BSET CORCON, #SATA ; Enable Saturation on Acc A
	; Init Pointers
	MOV #_ControlDifference, W8
	MOV #_PIDCoefficients, W10
	MOV #_RefSpeed, W1
	MOV #_Speed, W2
	MOV #_ControlOutput, W0
	; Calculate most recent error with saturation, no limit checking required
	LAC [W1], A ; A = RefSpeed
	LAC [W2], B ; B = Speed
	SUB A ; Diff = RefSpeed - Speed
	SAC A, [W8] ; ControlDifference[0] = A
	; Prepare MAC Operands
	;ControlDifference[i] * PIDCoefficients[i] to be used
	MOVSAC A, [W8]+=2, W4, [W10]+=2, W5
	LAC [W0], A ; Load Acc with last output
	; Perform MAC
	REPEAT #2	; Repeat 3 times
	MAC W4*W5, A, [W8]+=2, W4, [W10]+=2, W5
	; Store result in ControlOutput with saturation
	SAC A, [W0]
	BCLR CORCON, #SATA ; Disable Saturation on Acc A
	MOV #_ControlDifference, W8
	MOV [W8+2], W2 	;ControlDifference[2] = ControlDifference[1]
	MOV W2, [W8+4]
	MOV [W8], W2	;ControlDifference[1] = ControlDifference[0]
	MOV W2, [W8+2]
	RETURN;






;*********************************************************************
;                                                                    *
;                       Software License Agreement                   *
;                                                                    *
;   The software supplied herewith by Microchip Technology           *
;   Incorporated (the "Company") for its dsPIC controller            *
;   is intended and supplied to you, the Company's customer,         *
;   for use solely and exclusively on Microchip dsPIC                *
;   products. The software is owned by the Company and/or its        *
;   supplier, and is protected under applicable copyright laws. All  *
;   rights are reserved. Any use in violation of the foregoing       *
;   restrictions may subject the user to criminal sanctions under    *
;   applicable laws, as well as to civil liability for the breach of *
;   the terms and conditions of this license.                        *
;                                                                    *
;   THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION.  NO           *
;   WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING,    *
;   BUT NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND    *
;   FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE     *
;   COMPANY SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL,  *
;   INCIDENTAL OR CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.  *
;                                                                    *
;   (c) Copyright 2003 Microchip Technology, All rights reserved.    *
;*********************************************************************


; Local inclusions.
.nolist
.list

.equ    offsetpidCoefficients, 0
.equ    offseterrorHistory, 2
.equ    offsetiAccum, 4
.equ    offsetinputScale, 6
.equ    offsetcontrolOutput, 8
.equ 	offsetmeasuredOutput, 10
.equ 	offsetcontrolReference, 12

; Prepare CORCON for fractional computation.
.macro  fractsetup      wx              ; [
mov     #FRACT_MODE,\wx
mov     \wx,CORCON
.endm   ; ]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

 ;       .section .libdsp, code

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; _PID:
; Prototype:
;              tPID PID ( tPID *fooPIDStruct )
;
; Operation:
;
;                                             ----   Proportional
;                                            |    |  Output
;                             ---------------| Kp |-----------------
;                            |               |    |                 |
;                            |                ----                  |
;Reference                   |                                     ---
;Input         ---           |           --------------  Integral | + | Control   -------
;     --------| + |  Control |          |      Ki      | Output   |   | Output   |       |
;             |   |----------|----------| ------------ |----------|+  |----------| Plant |--
;        -----| - |Difference|          |              |          |   |          |       |  |
;       |      ---  (error)  |           --------------           | + |           -------   |
;       |                    |                                     ---                      |
;       | Measured           |         -------------------  Deriv   |                       |
;       | Outut              |        |                   | Output  |                       |
;       |                     --------| Kd                |---------                        |
;       |                             |                   |                                 |
;       |                              -------------------                                  |
;       |                                                                                   |
;       |                                                                                   |
;        -----------------------------------------------------------------------------------


;   controlHistory[n] = measuredOutput[n] - referenceInput[n]
;  where:
;   abcCoefficients, controlHistory, controlOutput, measuredOutput and controlReference
;   are all members of the data structure tPID.
;
; Input:
;       w0 = Address of tPID2 data structure

; Return:
;       w0 = Address of tPID2 data structure
;
; System resources usage:
;       {w0..w5}        used, not restored
;       {w9, w10}        saved, used, restored
;        AccA, AccB     used, not restored
;        CORCON         saved, used, restored
;
; DO and REPEAT instruction usage.
;       0 level DO instruction
;       0 REPEAT intructions
;
; Program words (24-bit instructions):
;       28
;
; Cycles (including C-function call and return overheads):
;       37 = 35 + 2 (C)  =  1.85 us @20MIPS
;............................................................................

        .global _PID2                    ; provide global scope to routine
_PID2:

        ; Save working registers.
		push 	w9
        push    w10
        push    CORCON                  

		BSET CORCON, #SATA ; Enable Saturation on Acc A
		BSET CORCON, #SATB ; Enable Saturation on Acc B

		;scale control reference and measured output values
		mov [w0 + #offsetinputScale], w5 ; w5 = inputScale
		mov [w0 + #offsetcontrolReference], w4 ; w4 = controlReference
		mpy w4*w5, a;
		mov [w0 + #offsetmeasuredOutput], w4 ; w4 = measuredOutput
		mpy w4*w5, b;
		sub	a; A = controlReference - measuredOutput = a  - b = error
				
		;integral accum with saturation
		mov [w0 + #offsetiAccum], w4 ; w4 = iAccum
		lac	w4,b ; b = iAccum
		add b; b = Itegral value, with saturation = a + b
		sac.r b,w4 ;w4 = iAccum
		mov w4, [w0 + #offsetiAccum]
		
		;error derivative
		;A still contains the error
		sac a, w2; w2 = saving actual scaled error 
		mov [w0 + #offseterrorHistory], w1 ;w1 = address of error history
		lac [w1], b; ; b = *error[0] = error from last iteration
		sub a; a = actual error - last error; a = a - b
		sac a, w3; w3 contains de/dt
		mov w2, [w1]; storing new error
		mov w3, [w1 + #2]; storing error derivative
		
		
		;Calculate PID output
		;clr a, 
 		mov [w0 + #offsetpidCoefficients], w9 ; w9 = address of pid coefficients
		mov [w0 + #offseterrorHistory], w10; w10 = address of error history containing e(t), d(e)/dt
		;Prepare mac operands
		movsac a, [w9], w4, [w10], w5; w4 = Kp, w5 = error
		mpy w4*w5, a,[w9]+=4,w4, [w10]+=2,w5 ; A = Kp * error
		mac w4*w5, a,[w9]-=2,w4, [w10], w5; A += Kd * de/dt
		mov [w0 + #offsetiAccum], w5 ;w5 = iAccum
		mac w4*w5, a,[w9],w4         ;A += Ki * iAccum

		;Store PID output
		sac.r	a, w1
		mov		w1,[w0 + #offsetcontrolOutput]

        pop     CORCON                  ; restore CORCON.
        pop     w10                     ; Restore working registers.
		pop		w9
        return

;END FILE
.end
