/**
 Copyright (C) 2009 IntRoLab
 http://introlab.gel.usherbrooke.ca
 Dominic Létourneau, ing. M.Sc.A.
 Dominic.Letourneau@USherbrooke.ca
 */
#ifndef _DRIVE_VIEW_H_
#define _DRIVE_VIEW_H_

#include "ui_DriveView.h"
#include <QInputDialog>
#include <limits.h>
#include <QTime>
#include <qwt_plot_curve.h>
#include <vector>


#define VECTOR_SIZE_MAX 500

#ifndef MAXFLOAT
#define MAXFLOAT 1000000
#endif

class DriveView : public QWidget, public Ui::DriveView
{
	Q_OBJECT;
	
	public :
	
	DriveView(QWidget * parent = NULL, int id = 0)
	: QWidget(parent), m_id(id)
	{
		setupUi(this);
		
		//First column
		connect(toolButton_CtrlMode,SIGNAL(clicked()),this,SLOT(CtrlModeClicked()));
		connect(toolButton_CtrlType,SIGNAL(clicked()),this,SLOT(CtrlTypeClicked()));
		connect(toolButton_SetPointSrc,SIGNAL(clicked()),this,SLOT(SetPointSrcClicked()));
		connect(toolButton_PosMesSrc,SIGNAL(clicked()),this,SLOT(PosMesSrcClicked()));
		connect(toolButton_SetPoint,SIGNAL(clicked()),this,SLOT(SetPointClicked()));
		connect(toolButton_SetPointMax,SIGNAL(clicked()),this,SLOT(SetPointMaxClicked()));
		connect(toolButton_SetPointMin,SIGNAL(clicked()),this,SLOT(SetPointMinClicked()));
		
		//Second column
		connect(toolButton_ADCOffset,SIGNAL(clicked()),this,SLOT(ADCOffsetClicked()));
		
		//Third column
		connect(toolButton_SpeedMax,SIGNAL(clicked()),this,SLOT(SpeedMaxClicked()));
		connect(toolButton_AccelerationStep,SIGNAL(clicked()),this,SLOT(AccelerationStepClicked()));
		connect(toolButton_CurrentOffset,SIGNAL(clicked()),this,SLOT(CurrentOffsetClicked()));
		
		//Fourth column
		connect(toolButton_CurrentLimit,SIGNAL(clicked()),this,SLOT(CurrentLimitClicked()));
		connect(toolButton_EncoderBias,SIGNAL(clicked()),this,SLOT(EncoderBiasClicked()));
		connect(toolButton_MotorBias,SIGNAL(clicked()),this,SLOT(MotorBiasClicked()));
		
		//Fifth column
		connect(toolButton_pid_Kp,SIGNAL(clicked()),this,SLOT(pid_KpClicked()));
		connect(toolButton_pid_Ki,SIGNAL(clicked()),this,SLOT(pid_KiClicked()));
		connect(toolButton_pid_Kd,SIGNAL(clicked()),this,SLOT(pid_KdClicked()));
		connect(toolButton_pid_error_max,SIGNAL(clicked()),this,SLOT(pid_error_maxClicked()));
		
	
		m_startTime = QTime::currentTime();
		
		m_positionCurve = new QwtPlotCurve("Position");
		m_positionCurve->setPen(QPen(Qt::red));
		m_positionCurve->attach(qwtPlot);
		
		m_setPointCurve= new QwtPlotCurve("SetPoint");
		m_setPointCurve->setPen(QPen(Qt::green));
		m_setPointCurve->attach(qwtPlot);
		
		m_speedCurve = new QwtPlotCurve("Speed");
		m_speedCurve->setPen(QPen(Qt::blue));
		m_speedCurve->attach(qwtPlot);
		
	}
	
	void clearGraph()
	{
		m_positionCurve->detach();
		m_setPointCurve->detach();
		m_speedCurve->detach();
		
		m_timeVals.resize(0);
		m_positionVals.resize(0);
		m_setPointVals.resize(0);
		m_speedVals.resize(0);
		
		qwtPlot->clear();
	}
	
	void pushGraphPosition(short Position, short Speed, short SetPoint)
	{
		
		double time = m_startTime.elapsed() / 1000.0;
		
		
		if (m_timeVals.size() > VECTOR_SIZE_MAX)
		{
			m_timeVals.erase(m_timeVals.begin());
			m_positionVals.erase(m_positionVals.begin());
			m_setPointVals.erase(m_setPointVals.begin());
			m_speedVals.erase(m_speedVals.begin());
		}
		
		m_timeVals.push_back(time);
		m_positionVals.push_back(Position);
		m_setPointVals.push_back(SetPoint);
		m_speedVals.push_back(Speed);
		
		m_positionCurve->setData(&m_timeVals[0],&m_positionVals[0],std::min(m_timeVals.size(),m_positionVals.size()));
		m_positionCurve->attach(qwtPlot);		
		m_setPointCurve->setData(&m_timeVals[0],&m_setPointVals[0],std::min(m_timeVals.size(),m_setPointVals.size()));
		m_setPointCurve->attach(qwtPlot);
		m_speedCurve->setData(&m_timeVals[0],&m_speedVals[0],std::min(m_timeVals.size(),m_speedVals.size()));
		m_speedCurve->attach(qwtPlot);
		
		//qwtPlot->setAxisScale(1,0.0,10.0,1.0);
		
		//Update plot
		qwtPlot->replot();
		
		
	}
	
signals:
	//First column
	void CtrlModeChanged(int value, int id);
	void CtrlTypeChanged(int value, int id);
	void SetPointSrcChanged(int value, int id);
	void PosMesSrcChanged(int value, int id);
	void SetPointChanged(int value, int id);
	void SetPointMaxChanged(int value, int id);
	void SetPointMinChanged(int value, int id);
	
	//Second column
	void ADCOffsetChanged(int value, int id);
	
	//Third column
	void SpeedMaxChanged(int value, int id);
	void AccelerationStepChanged(int value, int id);
	void CurrentOffsetChanged(int value, int id);
	
	//Fourth column
	void CurrentLimitChanged(int value, int id);
	void EncoderBiasChanged(int value, int id);
	void MotorBiasChanged(int value, int id);
	
	//Fifth column
	void pid_KpChanged(double value, int id);
	void pid_KiChanged(double value, int id);
	void pid_KdChanged(double value, int id);
	void pid_error_maxChanged(double value, int id);
	
	protected slots:
	
	void CtrlModeClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("CtrlMode Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter CtrlMode",lineEdit_CtrlMode->text().toInt(),0,3,1,&ok);
		
		if (ok)
		{
			emit CtrlModeChanged(value,m_id);
		}	
	}
	
	void CtrlTypeClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("CtrlType Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter CtrlType",lineEdit_CtrlType->text().toInt(),0,3,1,&ok);
		
		if (ok)
		{
			emit CtrlTypeChanged(value,m_id);
		}	
	}
	
	void SetPointSrcClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("SetPointSrc Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter SetPointSrc",lineEdit_SetPointSrc->text().toInt(),0,4,1,&ok);
		
		if (ok)
		{
			emit SetPointSrcChanged(value,m_id);
		}	
	}
	
	void PosMesSrcClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("PosMesSrc Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter PosMesSrc",lineEdit_PosMesSrc->text().toInt(),0,3,1,&ok);
		
		if (ok)
		{
			emit PosMesSrcChanged(value,m_id);
		}	
	}
	
	void SetPointClicked()
	{
		bool ok;

		int value = QInputDialog::getInt(this,QString("SetPoint Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter SetPoint",lineEdit_SetPoint->text().toInt(),0,4096,1,&ok);
		
		if (ok)
		{
			emit SetPointChanged(value,m_id);
		}	
	}
	
	void SetPointMaxClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("SetPointMax Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter SetPointMax",lineEdit_SetPointMax->text().toInt(),0,4096,1,&ok);
		
		if (ok)
		{
			emit SetPointMaxChanged(value,m_id);
		}	
	}
	
	void SetPointMinClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("SetPointMin Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter SetPointMin",lineEdit_SetPointMin->text().toInt(),0,4096,1,&ok);
		
		if (ok)
		{
			emit SetPointMinChanged(value,m_id);
		}	
	}
	
	void ADCOffsetClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("ADCOffset Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter ADCOffset",lineEdit_ADCOffset->text().toInt(),-4096,4096,1,&ok);
		
		if (ok)
		{
			emit ADCOffsetChanged(value,m_id);
		}	
	}
	
	void SpeedMaxClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("SpeedMax Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter SpeedMax",lineEdit_SpeedMax->text().toInt(),0,4096,1,&ok);
		
		if (ok)
		{
			emit SpeedMaxChanged(value,m_id);
		}	
	}
	
	void AccelerationStepClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("AccelerationStep Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter AccelerationStep",lineEdit_AccelerationStep->text().toInt(),0,4096,1,&ok);
		
		if (ok)
		{
			emit AccelerationStepChanged(value,m_id);
		}	
	}
	
	void CurrentOffsetClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("CurrentOffset Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter CurrentOffset",lineEdit_CurrentOffset->text().toInt(),0,4096,1,&ok);
		
		if (ok)
		{
			emit CurrentOffsetChanged(value,m_id);
		}	
	}
	
	
	void CurrentLimitClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("CurrentLimit Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter CurrentLimit",lineEdit_CurrentLimit->text().toInt(),0,10000,1,&ok);
		
		if (ok)
		{
			emit CurrentLimitChanged(value,m_id);
		}	
	}
	
	void EncoderBiasClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("EncoderBias Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter EncoderBias",lineEdit_EncoderBias->text().toInt(),0,1,1,&ok);
		
		if (ok)
		{
			emit EncoderBiasChanged(value,m_id);
		}	
	}
	
	void MotorBiasClicked()
	{
		bool ok;
		
		int value = QInputDialog::getInt(this,QString("MotorBias Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter MotorBias",lineEdit_MotorBias->text().toInt(),0,1,1,&ok);
		
		if (ok)
		{
			emit MotorBiasChanged(value,m_id);
		}	
	}
	
	void pid_KpClicked()
	{
		bool ok;
		
		double value = QInputDialog::getDouble(this,QString("pid_Kp Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter pid_Kp",lineEdit_pid_Kp->text().toDouble(),0,MAXFLOAT,6,&ok);
		
		if (ok)
		{
			emit pid_KpChanged(value,m_id);
		}	
	}
	
	void pid_KiClicked()
	{
		bool ok;
		
		double value = QInputDialog::getDouble(this,QString("pid_Ki Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter pid_Ki",lineEdit_pid_Ki->text().toDouble(),0,MAXFLOAT,6,&ok);
		
		if (ok)
		{
			emit pid_KiChanged(value,m_id);
		}	
	}
	
	void pid_KdClicked()
	{
		bool ok;
		
		double value = QInputDialog::getDouble(this,QString("pid_Kd Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter pid_Kd",lineEdit_pid_Kd->text().toDouble(),0,MAXFLOAT,6,&ok);
		
		if (ok)
		{
			emit pid_KdChanged(value,m_id);
		}	
	}
	
	void pid_error_maxClicked()
	{
		bool ok;
		
		double value = QInputDialog::getDouble(this,QString("pid_error_max Selection [Motor ID: ") + QString::number(m_id + 1) + "]","Enter pid_error_max",lineEdit_pid_error_max->text().toDouble(),0,MAXFLOAT,6,&ok);
		
		if (ok)
		{
			emit pid_error_maxChanged(value,m_id);
		}	
	}
	
	

	
	
protected:
	int m_id;
	QwtPlotCurve *m_positionCurve;
	QwtPlotCurve *m_setPointCurve;
	QwtPlotCurve *m_speedCurve;
	
	QTime m_startTime;
	std::vector<double> m_timeVals;
	std::vector<double> m_positionVals;
	std::vector<double> m_setPointVals;
	std::vector<double> m_speedVals;

};	

#endif
