/**
 Copyright (C) 2009 IntRoLab
 http://introlab.gel.usherbrooke.ca
 Dominic Létourneau, ing. M.Sc.A.
 Dominic.Letourneau@USherbrooke.ca
 */

#include "MainWindow.h"
#include <QFileDialog>

MainWindow::MainWindow(QWidget *parent)
: QMainWindow(parent), m_serialDriver(NULL), m_scriptTimer(NULL)
{
	setupUi(this);
	
	
	m_driveView[0] = new DriveView(tab_1,0);
	MotorLayout_1->addWidget(m_driveView[0]);
	m_driveView[1] = new DriveView(tab_2,1);
	MotorLayout_2->addWidget(m_driveView[1]);
	m_driveView[2] = new DriveView(tab_3,2);
	MotorLayout_3->addWidget(m_driveView[2]);
	m_driveView[3] = new DriveView(tab_4,3);
	MotorLayout_4->addWidget(m_driveView[3]);
	
	
	m_MLAWidget = new Ui::MLA();
	m_MLAWidget->setupUi(MLA_Widget);
	
	m_MRAWidget = new Ui::MRA();
	m_MRAWidget->setupUi(MRA_Widget);
	
	m_MRNWidget = new Ui::MRN();
	m_MRNWidget->setupUi(MRN_Widget);
	
	//Connect signals
	for (int i = 0; i < NB_DRIVE; i++)
	{
		connect(m_driveView[i],SIGNAL(CtrlModeChanged(int,int)),this,SLOT(CtrlModeChanged(int,int)));
		connect(m_driveView[i],SIGNAL(CtrlTypeChanged(int,int)),this,SLOT(CtrlTypeChanged(int,int)));
		connect(m_driveView[i],SIGNAL(SetPointSrcChanged(int,int)),this,SLOT(SetPointSrcChanged(int,int)));
		connect(m_driveView[i],SIGNAL(PosMesSrcChanged(int,int)),this,SLOT(PosMesSrcChanged(int,int)));
		connect(m_driveView[i],SIGNAL(SetPointChanged(int,int)),this,SLOT(SetPointChanged(int,int)));
		connect(m_driveView[i],SIGNAL(SetPointMaxChanged(int,int)),this,SLOT(SetPointMaxChanged(int,int)));
		connect(m_driveView[i],SIGNAL(SetPointMinChanged(int,int)),this,SLOT(SetPointMinChanged(int,int)));
		connect(m_driveView[i],SIGNAL(ADCOffsetChanged(int,int)),this,SLOT(ADCOffsetChanged(int,int)));
		connect(m_driveView[i],SIGNAL(SpeedMaxChanged(int,int)),this,SLOT(SpeedMaxChanged(int,int)));
		connect(m_driveView[i],SIGNAL(AccelerationStepChanged(int,int)),this,SLOT(AccelerationStepChanged(int,int)));
		connect(m_driveView[i],SIGNAL(CurrentOffsetChanged(int,int)),this,SLOT(CurrentOffsetChanged(int,int)));
		connect(m_driveView[i],SIGNAL(CurrentLimitChanged(int,int)),this,SLOT(CurrentLimitChanged(int,int)));
		connect(m_driveView[i],SIGNAL(EncoderBiasChanged(int,int)),this,SLOT(EncoderBiasChanged(int,int)));
		connect(m_driveView[i],SIGNAL(MotorBiasChanged(int,int)),this,SLOT(MotorBiasChanged(int,int)));
		connect(m_driveView[i],SIGNAL(pid_KpChanged(double,int)),this,SLOT(pid_KpChanged(double,int)));
		connect(m_driveView[i],SIGNAL(pid_KiChanged(double,int)),this,SLOT(pid_KiChanged(double,int)));
		connect(m_driveView[i],SIGNAL(pid_KdChanged(double,int)),this,SLOT(pid_KdChanged(double,int)));
		connect(m_driveView[i],SIGNAL(pid_error_maxChanged(double,int)),this,SLOT(pid_error_maxChanged(double,int)));
	}
	
	m_selectedDrive = 0x01;
	
	//Create serial port entries for WIN32
#ifdef WIN32
	serialComboBox->addItem("COM1");
	serialComboBox->addItem("COM2");
	serialComboBox->addItem("COM3");
	serialComboBox->addItem("COM4");
	serialComboBox->addItem("COM5");
	serialComboBox->addItem("COM6");
#endif
	
	//Create serial port entries for OSX    
#ifdef __APPLE__
	serialComboBox->addItem("/dev/tty.usbserial");
#endif
	
	//Create serial port entries for Linux  
#ifdef __linux__
	serialComboBox->addItem("/dev/ttyUSB0");
	serialComboBox->addItem("/dev/ttyUSB1");
	serialComboBox->addItem("/dev/ttyUSB2");
	serialComboBox->addItem("/dev/ttyUSB3");
	serialComboBox->addItem("/dev/ttyUSB4");
	serialComboBox->addItem("/dev/ttyUSB5");        
#endif  
	
	//MLA,MRA,MRN signals
	connect(m_MLAWidget->pushButton_Elbow,SIGNAL(clicked()),this,SLOT(MLA_elbowClicked()));
	connect(m_MLAWidget->pushButton_Pitch,SIGNAL(clicked()),this,SLOT(MLA_pitchClicked()));
	connect(m_MLAWidget->pushButton_Yaw,SIGNAL(clicked()),this,SLOT(MLA_yawClicked()));
	connect(m_MLAWidget->pushButton_Go,SIGNAL(clicked()),this,SLOT(MLA_goClicked()));
	connect(m_MRAWidget->pushButton_Elbow,SIGNAL(clicked()),this,SLOT(MRA_elbowClicked()));
	connect(m_MRAWidget->pushButton_Pitch,SIGNAL(clicked()),this,SLOT(MRA_pitchClicked()));
	connect(m_MRAWidget->pushButton_Yaw,SIGNAL(clicked()),this,SLOT(MRA_yawClicked()));
	connect(m_MRAWidget->pushButton_Go,SIGNAL(clicked()),this,SLOT(MRA_goClicked()));
	connect(m_MRNWidget->pushButton_Pitch,SIGNAL(clicked()),this,SLOT(MRN_pitchClicked()));
	connect(m_MRNWidget->pushButton_Yaw,SIGNAL(clicked()),this,SLOT(MRN_yawClicked()));
	connect(m_MRNWidget->pushButton_Go,SIGNAL(clicked()),this,SLOT(MRN_goClicked()));
	
	
	//Script signals
	connect(pushButton_ScriptLoad,SIGNAL(clicked()),this,SLOT(ScriptLoadClicked()));
	connect(pushButton_ScriptStart,SIGNAL(clicked()),this,SLOT(ScriptStartClicked()));
	connect(pushButton_ScriptStop, SIGNAL(clicked()),this,SLOT(ScriptStopClicked()));
	
	
	//Connect button
	connect(pushButton_Connect,SIGNAL(clicked()),this,SLOT(connectButtonClicked()));
	connect(pushButton_Disconnect,SIGNAL(clicked()),this,SLOT(disconnectButtonClicked()));
	
	//Save EEPROM
	connect(pushButton_SaveEEPROM,SIGNAL(clicked()),this,SLOT(SaveEEPROMClicked()));
	
	//ButtonBox button
	connect(buttonBox,SIGNAL(clicked(QAbstractButton*)),this,SLOT(buttonBoxButtonClicked(QAbstractButton*)));
	
	
	//Serial message handler
	m_txHandler = new QTimer(this);
	connect(m_txHandler,SIGNAL(timeout()),this,SLOT(handleTxMessage()));
	
#if WIN32
	m_txHandler->start(10);//10ms timer
#else
	m_txHandler->start(3);//3ms timer
#endif
}

void MainWindow::MLA_elbowClicked()
{
	bool ok;
	
	short value = QInputDialog::getInt(this,"MLA Elbow Value","Enter MLA Elbow value",m_MLAWidget->lineEdit_Elbow->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MLAWidget->lineEdit_Elbow->setText(QString::number(value));
	}	
	
}

void MainWindow::MLA_pitchClicked()
{
	
	bool ok;
	
	short value = QInputDialog::getInt(this,"MLA Pitch Value","Enter MLA Pitch value",m_MLAWidget->lineEdit_Pitch->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MLAWidget->lineEdit_Pitch->setText(QString::number(value));
	}	
}

void MainWindow::MLA_yawClicked()
{
	bool ok;
	
	short value = QInputDialog::getInt(this,"MLA Yaw Value","Enter MLA Yaw value",m_MLAWidget->lineEdit_Yaw->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MLAWidget->lineEdit_Yaw->setText(QString::number(value));
	}	
	
}

void MainWindow::MLA_goClicked()
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'l';
	Message.m_header[2] = 'a';
	
	bool ok;
	
	short elbow = m_MLAWidget->lineEdit_Elbow->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MLA - Elbow value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	short pitch = m_MLAWidget->lineEdit_Pitch->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MLA - Pitch value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	short yaw = m_MLAWidget->lineEdit_Yaw->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MLA - Yaw value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	Message.m_size = 9;
	Message.m_data[3] = elbow;
	Message.m_data[4] = elbow >> 8;
	Message.m_data[5] = yaw;
	Message.m_data[6] = yaw >> 8;
	Message.m_data[7] = pitch;
	Message.m_data[8] = pitch >> 8;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
}

void MainWindow::MRA_elbowClicked()
{
	bool ok;
	
	short value = QInputDialog::getInt(this,"MRA Elbow Value","Enter MRA Elbow value",m_MRAWidget->lineEdit_Elbow->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MRAWidget->lineEdit_Elbow->setText(QString::number(value));
	}	
	
}

void MainWindow::MRA_pitchClicked()
{
	bool ok;
	
	short value = QInputDialog::getInt(this,"MRA Pitch Value","Enter MRA Pitch value",m_MRAWidget->lineEdit_Pitch->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MRAWidget->lineEdit_Pitch->setText(QString::number(value));
	}	
	
}

void MainWindow::MRA_yawClicked()
{
	bool ok;
	
	short value = QInputDialog::getInt(this,"MRA Yaw Value","Enter MRA Yaw value",m_MRAWidget->lineEdit_Yaw->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MRAWidget->lineEdit_Yaw->setText(QString::number(value));
	}	
	
	
}

void MainWindow::MRA_goClicked()
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'r';
	Message.m_header[2] = 'a';
	
	bool ok;
	
	short elbow = m_MRAWidget->lineEdit_Elbow->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MRA - Elbow value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	short pitch = m_MRAWidget->lineEdit_Pitch->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MRA - Pitch value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	short yaw = m_MRAWidget->lineEdit_Yaw->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MRA - Yaw value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	Message.m_size = 9;
	Message.m_data[3] = elbow;
	Message.m_data[4] = elbow >> 8;
	Message.m_data[5] = yaw;
	Message.m_data[6] = yaw >> 8;
	Message.m_data[7] = pitch;
	Message.m_data[8] = pitch >> 8;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
	
	
}

void MainWindow::MRN_pitchClicked()
{
	bool ok;
	
	short value = QInputDialog::getInt(this,"MRN Pitch Value","Enter MRN Pitch value",m_MRNWidget->lineEdit_Pitch->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MRNWidget->lineEdit_Pitch->setText(QString::number(value));
	}	
	
}

void MainWindow::MRN_yawClicked()
{
	bool ok;
	
	short value = QInputDialog::getInt(this,"MRN Yaw Value","Enter MRN Yaw value",m_MRNWidget->lineEdit_Yaw->text().toInt(),0,4096,1,&ok);
	
	if (ok)
	{
		m_MRNWidget->lineEdit_Yaw->setText(QString::number(value));
	}	
	
}

void MainWindow::MRN_goClicked()
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'r';
	Message.m_header[2] = 'n';
	
	bool ok;
	
	
	short pitch = m_MRNWidget->lineEdit_Pitch->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MRN - Pitch value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	short yaw = m_MRNWidget->lineEdit_Yaw->text().toInt(&ok);
	if (!ok)
	{
		QMessageBox msgBox;
		msgBox.setText(QString("MRN - Yaw value invalid"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
		return;
	}
	
	Message.m_size = 7;
	Message.m_data[3] = pitch;
	Message.m_data[4] = pitch >> 8;
	Message.m_data[5] = yaw;
	Message.m_data[6] = yaw >> 8;
	
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
	
}

void MainWindow::SaveEEPROMClicked()
{
	unsigned char save = 1;
	unsigned int offset = ((char*) &m_globalVariables.m_WriteEEPROM) - ((char*)&m_globalVariables);
	sendVariableData(offset,save);
	
	QMessageBox msgBox;
	msgBox.setText(QString("Configuration saved to EEPROM for drive :")  + QString::number((int) m_selectedDrive));
	msgBox.setStandardButtons(QMessageBox::Ok);
	msgBox.setDefaultButton(QMessageBox::Ok);
	msgBox.exec();
}


void MainWindow::connectButtonClicked()
{
	if (m_serialDriver)
	{	
		delete m_serialDriver;
	}
	
	//Serial Port
	m_serialDriver = new SerialDriver(serialComboBox->currentText(),this);
	m_serialDriver->openPort();
	connect(m_serialDriver,SIGNAL(messageReady(const SerialMessage&)),this,SLOT(SerialMessageReady(const SerialMessage&)));
	labelConnected->setText("Connected");
	serialComboBox->setEnabled(false);
	pushButton_Connect->setEnabled(false);
	pushButton_Disconnect->setEnabled(true);
	driveSelectionBox->setEnabled(true);
	tabWidget->setEnabled(true);
	addressComboBox->setEnabled(true);
	
	connect(addressComboBox, SIGNAL(currentIndexChanged(const QString&)),this,SLOT(driveIndexChanged ( const QString &)));
	
	//Update Timer
	m_offset = 0;
	m_updateTimer = new QTimer(this);
	connect(m_updateTimer,SIGNAL(timeout()),this,SLOT(updateTimer()));
	m_updateTimer->start(50); //50ms timer
}

void MainWindow::disconnectButtonClicked()
{
	if (m_serialDriver)
	{
		delete m_serialDriver;
		m_serialDriver = NULL;
	}	
	
	labelConnected->setText("Disconnected");
	serialComboBox->setEnabled(true);
	pushButton_Connect->setEnabled(true);
	pushButton_Disconnect->setEnabled(false);
	driveSelectionBox->setEnabled(false);
	tabWidget->setEnabled(false);
	addressComboBox->setEnabled(false);
	
	//Timer
	m_offset = 0;
	delete m_updateTimer;
	m_updateTimer = NULL;
}

void MainWindow::driveIndexChanged ( const QString & text )
{
	//qDebug() << "driveIndexChanged ( const QString & text ) text = " << text;
	
	
	if (text == "0x01")
	{
		m_selectedDrive = 0x01;
	}
	else if (text == "0x02")
		
	{
		m_selectedDrive = 0x02;
	}
	
	//qDebug("Switching to drive : %i",(int) m_selectedDrive);
	
	for (int i = 0; i < NB_DRIVE; i++)
	{
		m_driveView[i]->clearGraph();
	}
}

void MainWindow::buttonBoxButtonClicked(QAbstractButton *button)
{
	qApp->quit();
}

void MainWindow::SerialMessageReady(const SerialMessage& message)
{
	if (message.msg_type == CAN_TYPE_REQUEST_DATA)
	{
		//qDebug("CAN_TYPE_REQUEST_DATA received");
		
		//hacking position offset
		unsigned int index = (message.msg_priority & 0x07);
		index = (index << 8);
		index |= message.msg_command;
		
		//qDebug("recv index : %i",index);
		
		for (int i = 0; i < (message.msg_boot_rtr_length & 0x0F); i++)
		{
			Q_ASSERT(index + i < sizeof(GlobalVariables));
			m_globalVariables.m_data[index + i] = message.msg_data[i];
		}
		
		//update GUI
		updateView();
	}
	else if (message.msg_type == CAN_TYPE_ACTUATOR_HIGH_PRIORITY)
	{
		
		switch(message.msg_command)
		{
				
			case CAN_ACTUATOR_CMD_POSITIONS:
				if ((message.msg_boot_rtr_length & 0x0F) == 8)
				{
					memcpy((char*) &m_globalVariables.m_variables[0].Position, &message.msg_data[0],2);
					memcpy((char*) &m_globalVariables.m_variables[1].Position, &message.msg_data[2],2);
					memcpy((char*) &m_globalVariables.m_variables[2].Position, &message.msg_data[4],2);
					memcpy((char*) &m_globalVariables.m_variables[3].Position, &message.msg_data[6],2);
					
					//Update graphs
					m_driveView[0]->pushGraphPosition(m_globalVariables.m_variables[0].Position, m_globalVariables.m_variables[0].Speed * 10, m_globalVariables.m_variables[0].SetPoint);
					m_driveView[1]->pushGraphPosition(m_globalVariables.m_variables[1].Position, m_globalVariables.m_variables[1].Speed * 10, m_globalVariables.m_variables[1].SetPoint);
					m_driveView[2]->pushGraphPosition(m_globalVariables.m_variables[2].Position, m_globalVariables.m_variables[2].Speed * 10, m_globalVariables.m_variables[2].SetPoint);
					m_driveView[3]->pushGraphPosition(m_globalVariables.m_variables[3].Position, m_globalVariables.m_variables[3].Speed * 10, m_globalVariables.m_variables[3].SetPoint);
					
				}	
				break;
				
			case CAN_ACTUATOR_CMD_SPEEDS:
				if ((message.msg_boot_rtr_length & 0x0F) == 8)
				{
					memcpy((char*) &m_globalVariables.m_variables[0].Speed, &message.msg_data[0],2);
					memcpy((char*) &m_globalVariables.m_variables[1].Speed, &message.msg_data[2],2);
					memcpy((char*) &m_globalVariables.m_variables[2].Speed, &message.msg_data[4],2);
					memcpy((char*) &m_globalVariables.m_variables[3].Speed, &message.msg_data[6],2);
				}	
				break;
				
			case CAN_ACTUATOR_CMD_SETPOINTS:
				if ((message.msg_boot_rtr_length & 0x0F) == 8)
				{
					memcpy((char*) &m_globalVariables.m_variables[0].SetPoint, &message.msg_data[0],2);
					memcpy((char*) &m_globalVariables.m_variables[1].SetPoint, &message.msg_data[2],2);
					memcpy((char*) &m_globalVariables.m_variables[2].SetPoint, &message.msg_data[4],2);
					memcpy((char*) &m_globalVariables.m_variables[3].SetPoint, &message.msg_data[6],2);
				}	
				break;
		}
	}	
	
	
}	

void MainWindow::handleTxMessage()
{
	if (m_serialDriver)
	{
		if (!m_serialQueue.isEmpty())
		{
			SerialMessage msg = m_serialQueue.front();
			m_serialQueue.pop_front();
			m_serialDriver->writeMessage(msg);
		}
	}
	else
	{
		m_serialQueue.clear();
	}
}

void MainWindow::updateTimer()
{
	CANMessage canMessage;
	
	
	//Send request for RAM	
	canMessage.msg_priority = (m_offset >> 8);
	canMessage.msg_type = CAN_TYPE_REQUEST_DATA;
	canMessage.msg_remote = 1;
	canMessage.msg_data_length = 8;
	canMessage.msg_cmd = m_offset;
	canMessage.msg_eeprom_ram = CAN_REQUEST_RAM;
	canMessage.msg_read_write = CAN_REQUEST_READ;
	canMessage.msg_dest = m_selectedDrive;
	
	//Send serial message
	sendCANMessage(canMessage);
	
	//prepare for next request
	//Warning : this works because sizeof(GlobalVariables) is a factor of 8
	if (m_offset + 8 >= sizeof(GlobalVariables))
	{
		m_offset = 0;
	}
	else
	{
		m_offset += 8;
	}
	
	//Ask for periodic positions
	canMessage.msg_priority = 0;
	canMessage.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY;
	canMessage.msg_remote = 1;
	canMessage.msg_data_length = 8;
	canMessage.msg_cmd = CAN_ACTUATOR_CMD_POSITIONS;
	canMessage.msg_eeprom_ram = 1;
	canMessage.msg_read_write = 1;
	canMessage.msg_dest = m_selectedDrive;
	
	//Send serial message
	sendCANMessage(canMessage);
	
	
	//Ask for periodic setpoints
	canMessage.msg_priority = 0;
	canMessage.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY;
	canMessage.msg_remote = 1;
	canMessage.msg_data_length = 8;
	canMessage.msg_cmd = CAN_ACTUATOR_CMD_SETPOINTS;
	canMessage.msg_eeprom_ram = 1;
	canMessage.msg_read_write = 1;
	canMessage.msg_dest = m_selectedDrive;
	
	//Send serial message
	sendCANMessage(canMessage);
	
	//Ask for periodic speeds
	canMessage.msg_priority = 0;
	canMessage.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY;
	canMessage.msg_remote = 1;
	canMessage.msg_data_length = 8;
	canMessage.msg_cmd = CAN_ACTUATOR_CMD_SPEEDS;
	canMessage.msg_eeprom_ram = 1;
	canMessage.msg_read_write = 1;
	canMessage.msg_dest = m_selectedDrive;
	
	//Send serial message
	sendCANMessage(canMessage);
	
}

void MainWindow::updateView()
{
	//qDebug("updateView()");
	
	for (int i = 0; i < NB_DRIVE; i++)
	{	
		//Group 1
		m_driveView[i]->lineEdit_CtrlMode->setText(QString::number(m_globalVariables.m_variables[i].CtrlMode));
		m_driveView[i]->lineEdit_CtrlType->setText(QString::number(m_globalVariables.m_variables[i].CtrlType));
		m_driveView[i]->lineEdit_SetPointSrc->setText(QString::number(m_globalVariables.m_variables[i].SetPointSource));
		m_driveView[i]->lineEdit_PosMesSrc->setText(QString::number(m_globalVariables.m_variables[i].PosMesSource));
		m_driveView[i]->lineEdit_ErrorCode->setText(QString::number(m_globalVariables.m_variables[i].ErrorCode));
		m_driveView[i]->lineEdit_SetPoint->setText(QString::number(m_globalVariables.m_variables[i].SetPoint));
		m_driveView[i]->lineEdit_SetPointMax->setText(QString::number(m_globalVariables.m_variables[i].SetPointMax));
		m_driveView[i]->lineEdit_SetPointMin->setText(QString::number(m_globalVariables.m_variables[i].SetPointMin));
		
		//Group 2
		m_driveView[i]->lineEdit_ADCValue->setText(QString::number(m_globalVariables.m_variables[i].ADCValue));
		m_driveView[i]->lineEdit_ADCOffset->setText(QString::number(m_globalVariables.m_variables[i].ADCOffset));
		m_driveView[i]->lineEdit_ThermalState->setText(QString::number(m_globalVariables.m_variables[i].ThermalState));
		m_driveView[i]->lineEdit_Speed->setText(QString::number(m_globalVariables.m_variables[i].Speed));
		m_driveView[i]->lineEdit_Position->setText(QString::number(m_globalVariables.m_variables[i].Position));
		m_driveView[i]->lineEdit_Acceleration->setText(QString::number(m_globalVariables.m_variables[i].Acceleration));
		m_driveView[i]->lineEdit_RefPoint->setText(QString::number(m_globalVariables.m_variables[i].RefPoint));
		m_driveView[i]->lineEdit_MesPoint->setText(QString::number(m_globalVariables.m_variables[i].MesPoint));
		
		//Group 3
		m_driveView[i]->lineEdit_PIDOut->setText(QString::number(m_globalVariables.m_variables[i].PIDOut));
		m_driveView[i]->lineEdit_SpeedMax->setText(QString::number(m_globalVariables.m_variables[i].SpeedMax));
		m_driveView[i]->lineEdit_AccelerationStep->setText(QString::number(m_globalVariables.m_variables[i].AccelerationStep));
		m_driveView[i]->lineEdit_InitPoint->setText(QString::number(m_globalVariables.m_variables[i].InitPoint));
		m_driveView[i]->lineEdit_DestPoint->setText(QString::number(m_globalVariables.m_variables[i].DestPoint));
		m_driveView[i]->lineEdit_Current->setText(QString::number(m_globalVariables.m_variables[i].Current));
		m_driveView[i]->lineEdit_CurrentOffset->setText(QString::number(m_globalVariables.m_variables[i].CurrentOffset));
		
		//Group 4
		m_driveView[i]->lineEdit_NextPoint->setText(QString::number(m_globalVariables.m_variables[i].NextPoint));
		m_driveView[i]->lineEdit_CurrentLimit->setText(QString::number(m_globalVariables.m_variables[i].CurrentLimit));
		m_driveView[i]->lineEdit_PWM_CLimit->setText(QString::number(m_globalVariables.m_variables[i].PWM_CurrentLimit));
		m_driveView[i]->lineEdit_PWM_CurrentStep->setText(QString::number(m_globalVariables.m_variables[i].PWM_CurrentStep));
		m_driveView[i]->lineEdit_CurrentLimitActive->setText(QString::number(m_globalVariables.m_variables[i].CurrentLimitActive));
		m_driveView[i]->lineEdit_EncoderBias->setText(QString::number(m_globalVariables.m_variables[i].EncoderBias));
		m_driveView[i]->lineEdit_MotorBias->setText(QString::number(m_globalVariables.m_variables[i].MotorBias));
		
		//Group 5
		m_driveView[i]->lineEdit_pid_Kp->setText(QString::number(m_globalVariables.m_variables[i].pid_kp));
		m_driveView[i]->lineEdit_pid_Ki->setText(QString::number(m_globalVariables.m_variables[i].pid_ki));
		m_driveView[i]->lineEdit_pid_Kd->setText(QString::number(m_globalVariables.m_variables[i].pid_kd));
		m_driveView[i]->lineEdit_pid_Error->setText(QString::number(m_globalVariables.m_variables[i].pid_error));
		m_driveView[i]->lineEdit_pid_error_accum->setText(QString::number(m_globalVariables.m_variables[i].pid_error_accum));
		m_driveView[i]->lineEdit_pid_error_der->setText(QString::number(m_globalVariables.m_variables[i].pid_error_derivative));
		m_driveView[i]->lineEdit_pid_error_max->setText(QString::number(m_globalVariables.m_variables[i].pid_error_accum_max));	
		
		//Loop time
		lineEdit_LoopTime->setText(QString::number(m_globalVariables.m_loopTime));
		
	}
	
}

void MainWindow::sendCANMessage(const CANMessage& message)
{
	if (m_serialDriver)
	{
		SerialMessage serialMessage(message);
		m_serialQueue.push_back(serialMessage);
	}
}

void MainWindow::CtrlModeChanged(int value, int id)
{
	SharedVariables vars;
	vars.CtrlMode = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.CtrlMode -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.CtrlMode);
}

void MainWindow::CtrlTypeChanged(int value, int id)
{
	SharedVariables vars;
	vars.CtrlType = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.CtrlType -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.CtrlType);
}

void MainWindow::SetPointSrcChanged(int value, int id)
{
	SharedVariables vars;
	vars.SetPointSource = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.SetPointSource -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.SetPointSource);
}

void MainWindow::PosMesSrcChanged(int value, int id)
{
	SharedVariables vars;
	vars.PosMesSource = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.PosMesSource -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.PosMesSource);
}	

void MainWindow::SetPointChanged(int value, int id)
{
	SharedVariables vars;
	vars.SetPoint = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.SetPoint -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.SetPoint);	
}

void MainWindow::SetPointMaxChanged(int value, int id)
{
	SharedVariables vars;
	vars.SetPointMax = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.SetPointMax -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.SetPointMax);
}

void MainWindow::SetPointMinChanged(int value, int id)
{
	SharedVariables vars;
	vars.SetPointMin = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.SetPointMin -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.SetPointMin);
}

void MainWindow::ADCOffsetChanged(int value, int id)
{
	SharedVariables vars;
	vars.ADCOffset = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.ADCOffset -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.ADCOffset);
}

void MainWindow::SpeedMaxChanged(int value, int id)
{
	SharedVariables vars;
	vars.SpeedMax = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.SpeedMax -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.SpeedMax);
}	

void MainWindow::AccelerationStepChanged(int value, int id)
{
	SharedVariables vars;
	vars.AccelerationStep = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.AccelerationStep -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.AccelerationStep);
}

void MainWindow::CurrentOffsetChanged(int value, int id)
{
	SharedVariables vars;
	vars.CurrentOffset = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.CurrentOffset -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.CurrentOffset);
}

void MainWindow::CurrentLimitChanged(int value, int id)
{
	SharedVariables vars;
	vars.CurrentLimit = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.CurrentLimit -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.CurrentLimit);
}	

void MainWindow::EncoderBiasChanged(int value, int id)
{
	SharedVariables vars;
	vars.EncoderBias = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.EncoderBias -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.EncoderBias);
}

void MainWindow::MotorBiasChanged(int value, int id)
{
	SharedVariables vars;
	vars.MotorBias = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.MotorBias -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.MotorBias);
	
}

void MainWindow::pid_KpChanged(double value, int id)
{
	SharedVariables vars;
	vars.pid_kp = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.pid_kp -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.pid_kp);
}

void MainWindow::pid_KiChanged(double value, int id)
{
	SharedVariables vars;
	vars.pid_ki = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.pid_ki -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.pid_ki);
}

void MainWindow::pid_KdChanged(double value, int id)
{
	SharedVariables vars;
	vars.pid_kd = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.pid_kd -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.pid_kd);
}

void MainWindow::pid_error_maxChanged(double value, int id)
{
	SharedVariables vars;
	vars.pid_error_accum_max = value;
	unsigned int offset = id * sizeof(SharedVariables) + ((char*) &vars.pid_error_accum_max -(char*) &vars.CtrlMode);
	sendVariableData(offset, vars.pid_error_accum_max);
}


void MainWindow::sendVariableData(unsigned int offset, unsigned char value)
{
	
	//qDebug("sendVariableData(unsigned int offset = %i, unsigned char value = %i",offset,(int) value);
	
	CANMessage canMessage;
	
	
	//Send request for RAM	
	canMessage.msg_priority = (offset >> 8);
	canMessage.msg_type = CAN_TYPE_REQUEST_DATA;
	canMessage.msg_remote = 0;
	canMessage.msg_data_length = sizeof(value);
	canMessage.msg_cmd = offset;
	canMessage.msg_eeprom_ram = CAN_REQUEST_RAM;
	canMessage.msg_read_write = CAN_REQUEST_WRITE;
	canMessage.msg_dest = m_selectedDrive;
	memcpy(&canMessage.msg_data[0],(char*) &value, sizeof(value));
	
	//Send serial message
	sendCANMessage(canMessage);
	
}

void MainWindow::sendVariableData(unsigned int offset, short value)
{
	//qDebug("sendVariableData(unsigned int offset = %i, short value = %i",offset,(int) value);
	
	CANMessage canMessage;
	
	
	//Send request for RAM	
	canMessage.msg_priority = (offset >> 8);
	canMessage.msg_type = CAN_TYPE_REQUEST_DATA;
	canMessage.msg_remote = 0;
	canMessage.msg_data_length = sizeof(value);
	canMessage.msg_cmd = offset;
	canMessage.msg_eeprom_ram = CAN_REQUEST_RAM;
	canMessage.msg_read_write = CAN_REQUEST_WRITE;
	canMessage.msg_dest = m_selectedDrive;
	memcpy(&canMessage.msg_data[0],(char*) &value, sizeof(value));
	
	//Send serial message
	sendCANMessage(canMessage);
	
}

void MainWindow::sendVariableData(unsigned int offset, float value)
{
	//qDebug("sendVariableData(unsigned int offset = %i, float value = %f",offset,value);
	
	CANMessage canMessage;
	
	
	//Send request for RAM	
	canMessage.msg_priority = (offset >> 8);
	canMessage.msg_type = CAN_TYPE_REQUEST_DATA;
	canMessage.msg_remote = 0;
	canMessage.msg_data_length = sizeof(value);
	canMessage.msg_cmd = offset;
	canMessage.msg_eeprom_ram = CAN_REQUEST_RAM;
	canMessage.msg_read_write = CAN_REQUEST_WRITE;
	canMessage.msg_dest = m_selectedDrive;
	memcpy(&canMessage.msg_data[0],(char*) &value, sizeof(value));
	
	//Send serial message
	sendCANMessage(canMessage);
	
}


void MainWindow::ScriptLoadClicked()
{
		
	QString fileName = QFileDialog::getOpenFileName(this, tr("Open Script File"),
													"scripts/",
													tr("Script (*.txt)"));
	
	if (fileName.size())
	{
		QFile myFile(fileName);
		if (myFile.open(QIODevice::ReadOnly))
		{
			//Empty Table if already loaded
			while(tableWidget_Script->rowCount())
			{
				tableWidget_Script->removeRow(0); 
			}
			
			
			//Load semicolon separated values
			while(myFile.bytesAvailable())
			{
				QByteArray line = myFile.readLine();
				
				if (line.size() > 0 && line[0] != '#')
				{
					QStringList myList = QString(line).split(";");
					int row = tableWidget_Script->rowCount();
					tableWidget_Script->insertRow(row);
					for (int i = 0; i < myList.size(); i++)
					{
						tableWidget_Script->setItem(row,i,new QTableWidgetItem(myList[i]));
					}
				}
			}
			
			
			//Enable Start and Stop buttons
			pushButton_ScriptStart->setEnabled(true);
			pushButton_ScriptStop->setEnabled(true);
			
		}
		
	}
	
}

void MainWindow::ScriptStartClicked()
{
	if (m_scriptTimer)
	{
		delete m_scriptTimer;
	}	
	
	//Reset Index
	m_scriptIndex = 0;
	
	//Create timer
	m_scriptTimer = new QTimer(this);
	
	//Connect Signals
	connect(m_scriptTimer,SIGNAL(timeout()),this,SLOT(ScriptTimeout()));
	
	//Start Timer (100ms)
	m_scriptTimer->start(100);
}

void MainWindow::ScriptStopClicked()
{
	
	if (m_scriptTimer)
	{
		delete m_scriptTimer;
		m_scriptTimer = NULL;
		m_scriptIndex = 0;
	}
}	

void MainWindow::ScriptMLA(short elbow, short yaw, short pitch)
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'l';
	Message.m_header[2] = 'a';
	Message.m_data[3] = elbow;
	Message.m_data[4] = elbow >> 8;
	Message.m_data[5] = yaw;
	Message.m_data[6] = yaw >> 8;
	Message.m_data[7] = pitch;
	Message.m_data[8] = pitch >> 8;
	Message.m_size = 9;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
}

void MainWindow::ScriptMRA(short elbow, short yaw, short pitch)
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'r';
	Message.m_header[2] = 'a';
	Message.m_data[3] = elbow;
	Message.m_data[4] = elbow >> 8;
	Message.m_data[5] = yaw;
	Message.m_data[6] = yaw >> 8;
	Message.m_data[7] = pitch;
	Message.m_data[8] = pitch >> 8;
	Message.m_size = 9;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
}

void MainWindow::ScriptMRN(short pitch, short yaw)
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'r';
	Message.m_header[2] = 'n';
	Message.m_data[3] = pitch;
	Message.m_data[4] = pitch >> 8;
	Message.m_data[5] = yaw;
	Message.m_data[6] = yaw >> 8;
	Message.m_size = 7;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
}


void MainWindow::ScriptMRE(char le_yaw, char le_yaw_speed, char re_yaw, char re_yaw_speed, char pitch, char pitch_speed)
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'r';
	Message.m_header[2] = 'e';
	Message.m_data[3] = le_yaw;
	Message.m_data[4] = le_yaw_speed;
	Message.m_data[5] = re_yaw;
	Message.m_data[6] = re_yaw_speed;
	Message.m_data[7] = pitch;
	Message.m_data[8] = pitch_speed;
	Message.m_size = 9;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
}

void MainWindow::ScriptMEB(char left_roll, char left_roll_speed, char right_roll, char right_roll_speed)
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'e';
	Message.m_header[2] = 'b';
	Message.m_data[3] = left_roll;
	Message.m_data[4] = left_roll_speed;
	Message.m_data[5] = right_roll;
	Message.m_data[6] = right_roll_speed;
	Message.m_size = 7;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
}

void MainWindow::ScriptMRM(char ul_pos, char ul_speed, char ll_pos, char ll_speed, char ur_pos, char ur_speed, char lr_pos, char lr_speed)
{
	SerialMessage Message;
	Message.m_header[0] = 'm';
	Message.m_header[1] = 'r';
	Message.m_header[2] = 'm';
	Message.m_data[3] = ul_pos;
	Message.m_data[4] = ul_speed;
	Message.m_data[5] = ll_pos;
	Message.m_data[6] = ll_speed;
	Message.m_data[7] = ur_pos;
	Message.m_data[8] = ur_speed;
	Message.m_data[9] = lr_pos;
	Message.m_data[10] = lr_speed;
	Message.m_size = 11;
	
	//Enqueue message to send...
	m_serialQueue.push_back(Message);
	
}

void MainWindow::ScriptTimeout()
{
	if (m_scriptIndex < tableWidget_Script->rowCount())
	{
		tableWidget_Script->selectRow(m_scriptIndex);
		QStringList itemList;
		
		//Load non empty cells
		for (int i = 0; i < tableWidget_Script->columnCount(); i++)
		{
			QTableWidgetItem *item = tableWidget_Script->item(m_scriptIndex,i);
			if (item)
			{
				itemList.push_back(item->text());
			}
			else
			{
				break;
			}
		}	
		
		//Make sure the list is non empty
		if (itemList.size())
		{
			//Handle "wait"
			if(itemList[0] == "wait" && itemList.size() == 2)
			{
				m_scriptTimer->start(itemList[1].toInt());
			}
			else
			{
				//Default timing
				m_scriptTimer->start(100);
			}
			
			//Handle "mrm"
			if(itemList[0] == "mrm" && itemList.size() == 9)
			{
				//void MainWindow::ScriptMRM(char ul_pos, char ul_speed, char ll_pos, char ll_speed, char ur_pos, char ur_speed, char lr_pos, char lr_speed)
				ScriptMRM(itemList[1].toInt(), itemList[2].toInt(), itemList[3].toInt(), itemList[4].toInt(), itemList[5].toInt(), itemList[6].toInt(), itemList[7].toInt(), itemList[8].toInt());
			}
			else if (itemList[0] == "mrm")
			{
				qDebug("Script Error missing arguments for mrm command");
			}
			
			
			//Handle "mre"
			if(itemList[0] == "mre" && itemList.size() == 7)
			{
				//void MainWindow::ScriptMRE(char le_yaw, char le_yaw_speed, char re_yaw, char re_yaw_speed, char pitch, char pitch_speed)
				ScriptMRE(itemList[1].toInt(), itemList[2].toInt(), itemList[3].toInt(), itemList[4].toInt(), itemList[5].toInt(), itemList[6].toInt());
			}
			else if (itemList[0] == "mre")
			{
				qDebug("Script Error missing arguments for mre command");
			}
			
			//Handle "meb"
			if(itemList[0] == "meb" && itemList.size() == 5)
			{
				//void MainWindow::ScriptMEB(char left_roll, char left_roll_speed, char right_roll, char right_roll_speed)
				ScriptMEB(itemList[1].toInt(), itemList[2].toInt(), itemList[3].toInt(), itemList[4].toInt());
			}
			else  if (itemList[0] == "meb")
			{
				qDebug("Script Error missing arguments for meb command");
			}
			
			//Handle "mla"
			if(itemList[0] == "mla" && itemList.size() == 4)
			{
				ScriptMLA(itemList[1].toInt(),itemList[2].toInt(),itemList[3].toInt());
			}
			else  if (itemList[0] == "mla")
			{
				qDebug("Script Error missing arguments for mla command");
			}
			
			//Handle "mra"
			if(itemList[0] == "mra" && itemList.size() == 4)
			{
				ScriptMRA(itemList[1].toInt(),itemList[2].toInt(),itemList[3].toInt());
			}
			else if (itemList[0] == "mra")
			{
				qDebug("Script Error missing arguments for mra command");
			}
			
			//Handle "mrn"
			if(itemList[0] == "mrn" && itemList.size() == 3)
			{
				ScriptMRN(itemList[1].toInt(),itemList[2].toInt());
			}
			else if (itemList[0] == "mrn")
			{
				qDebug("Script Error missing arguments for mrn command");
			}
			
			//Handle "end"
			if (itemList[0] == "end")
			{
				m_scriptTimer->stop();
				QMessageBox msgBox;
				msgBox.setText(QString("Script Completed"));
				msgBox.setStandardButtons(QMessageBox::Ok);
				msgBox.setDefaultButton(QMessageBox::Ok);
				msgBox.exec();
			}
			
		}
		
		//Prepare next index
		m_scriptIndex++;
	}
	else
	{
		tableWidget_Script->selectRow(m_scriptIndex);
		m_scriptTimer->stop();
		m_scriptIndex=0;
		QMessageBox msgBox;
		msgBox.setText(QString("Script Completed"));
		msgBox.setStandardButtons(QMessageBox::Ok);
		msgBox.setDefaultButton(QMessageBox::Ok);
		msgBox.exec();
	}
	
	
}


