/**
 Copyright (C) 2009 IntRoLab
 http://introlab.gel.usherbrooke.ca
 Dominic Létourneau, ing. M.Sc.A.
 Dominic.Letourneau@USherbrooke.ca
 */

#include "SerialDriver.h"

SerialDriver::SerialDriver(QString comPort, QObject *parent)
:	QObject(parent), m_comPort(comPort), m_serialPort(NULL)
{
	m_rxBytes = 0;
	m_txBytes = 0;
	m_errorBytes = 0;
}

void SerialDriver::openPort()
{
	m_portSettings.BaudRate = BAUD57600;
	m_portSettings.DataBits =  DATA_8;
	m_portSettings.FlowControl = FLOW_OFF;
	m_portSettings.Parity = PAR_NONE;
	m_portSettings.StopBits = STOP_1;
	m_portSettings.Timeout_Millisec = 0;
	
	
	//Let's open the serial interface
	//Need to open port in polling mode for POSIX systems
#ifdef _TTY_POSIX_	
	m_serialPort = new QextSerialPort(m_comPort, m_portSettings, QextSerialBase::Polling);
#else			
	m_serialPort = new QextSerialPort(m_comPort, m_portSettings, QextSerialBase::EventDriven);
#endif	
	
	
	//OpenPort
	if (m_serialPort->open(QIODevice::ReadWrite))
	{	
		connect(m_serialPort,SIGNAL(aboutToClose()),this,SLOT(aboutToClose()));
		
		//Connect signals
#ifdef _TTY_POSIX_
		m_readTimer = new QTimer(this);
		
		//Useful for polling
		connect(m_readTimer,SIGNAL(timeout()),this,SLOT(readyRead()));
		m_readTimer->start(1); //1ms
#else				
		connect(m_serialPort,SIGNAL(readyRead()),this,SLOT(readyRead()));
#endif	
		
		//Emit connected
		emit connected();
		
		//Reset counter
		m_rxBytes = 0;
		m_txBytes = 0;
		m_errorBytes = 0;
		
		
		//Start test timer
		//m_testTimer = new QTimer(this);
		//connect(m_testTimer,SIGNAL(timeout()),this,SLOT(testTimer()));
		//m_testTimer->start(100);
		
	}
	else
	{
		emit disconnected();
	}	
	
}


bool SerialDriver::writeMessage(const SerialMessage& message)
{
		
	m_txBytes += message.m_size;
	message.write(*m_serialPort);
	return m_serialPort->waitForBytesWritten(10);

}


void SerialDriver::readyRead()
{
	
	if(m_serialPort->bytesAvailable() > 0)
	{	
			
		//qDebug("readyReady() %i bytes",(int)m_serialPort->bytesAvailable());
		QByteArray data = m_serialPort->readAll();
		m_rxBytes += data.size(); 
		parseData(data);
		
	}	
}

void SerialDriver::aboutToClose()
{
	qDebug("SerialDriver::aboutToClose()");
	emit disconnected();
}

void SerialDriver::parseData(QByteArray &data)
{
	
	
#ifdef _PROTOCOL_DEBUG_			
	qDebug("SerialDriver::parseData(QByteArray &data)");
#endif
	
	//Accumulate data
	m_buffer += data;
	
	
	while (m_buffer.size() >= MESSAGE_SIZE)
	{
		//Look for "can" header
		int pos = m_buffer.indexOf(QByteArray("can"));
		
		//qDebug("found pos : %i",pos);
			
		if (pos >= 0 )
		{
			if (pos == 0)
			{
				//Serial message OK!
				//qDebug("serial message ok");
			
				SerialMessage message;
				
				for (int i = 0; i< MESSAGE_SIZE; i++)
				{
					message.m_data[i] = m_buffer[i];
				}
				
				//message.print();
				
				emit messageReady(message);
				
				m_buffer.remove(0,MESSAGE_SIZE);
				
			}	
			else
			{
				//Cut the crap!
				//And start again
				m_buffer.remove(0,pos);
				continue;
			}	
		}
		else
		{
			//header not found, discarding...
			m_buffer.clear();
		}	
    }
} //parseData



void SerialDriver::testTimer()
{
		if(m_serialPort)
		{
			CANMessage canMessage;
			
			canMessage.msg_priority = 0;
			canMessage.msg_type = CAN_TYPE_REQUEST_DATA;
			canMessage.msg_remote = 0;
			canMessage.msg_data_length = 8;
			canMessage.msg_cmd = 0;
			canMessage.msg_eeprom_ram = CAN_REQUEST_RAM;
			canMessage.msg_read_write = CAN_REQUEST_WRITE;
			for (int i = 0; i < 8; i++)
			{	
				canMessage.msg_data[i] = i;
			}	
			
			canMessage.msg_dest = 0x01;
			
			SerialMessage serialMessage(canMessage);
			serialMessage.write(*m_serialPort);
		}
}



