//////////////////////////////////////////////////////////////////////
//   can18_driver.c
//////////////////////////////////////////////////////////////////////
//
//   Description: Driver construit spcialement pour les 18xxx8 qui
//                possde le can unsigned chargr.
//
//   Nom: Pierre Lepage
//   Date de cration : 2004/06/24
//
//   Histoire des modifications:
//
//   Nom                   Date          Descrition
//	 Dominic Letourneau		13 Jan 2005	 Porter le code pour C18
//
//////////////////////////////////////////////////////////////////////
//               Copyright by Laborius  2005
//////////////////////////////////////////////////////////////////////
#include <p18F458.h>
#include <delays.h>
#include <string.h>

// Library call
#include "can18_driver.h"

//macros
#define can_kbhit()                 (RXB0CON.rxful || RXB1CON.rxful)
#define can_tbe()                   (!TXB0CON.txreq || !TXB1CON.txreq || !TXB2CON.txreq)
#define can_abort()                 (CANCON.abat=1)

// GLOBAL VARIABLES TO BE USED IN ALL SOURCE FILES
LABORIUS_MESSAGE g_rMessage = {0};    // all downloaded messages from PIC
                                      // are written here
unsigned char data_flow_buffer[8];

volatile unsigned char DATA_FLOW_TABLE[128] = {0xFF};

//unsigned char Can_Addr = 0xFF;
unsigned char EEPROM_RD = 0;
unsigned char EEPROM_WR = 0;
unsigned char EEPROM_SECTION_ID = 0;
unsigned char MSG_SILENCE = 0;			//0 = Driver will send all msg ; 1 = Driver will not send msg except for the IM_ALIVE signal

/*******************************************************************
READ EEPROM
 *******************************************************************/
unsigned char can_read_eeprom(unsigned char addr) 
{ 
	volatile unsigned char eepTemp; 	
	EEADR = addr; 
	EECON1bits.EEPGD = 0; 
	EECON1bits.CFGS = 0; 
	EECON1bits.RD = 1; 
	Delay10TCYx(10);//10us delay
	eepTemp = EEDATA; 
	return eepTemp; 
} 

/*******************************************************************
WRITE EEPROM
 *******************************************************************/
void can_write_eeprom(unsigned char addr, unsigned char data) 
{ 	
	EECON1bits.EEPGD = 0; //EEPROM ACCESS
	EECON1bits.CFGS = 0;  //FLASH OR EEPROM
	EECON1bits.WREN = 1;  //WRITE ENABLE
	EEADR = addr; //SET ADDRESS
	EEDATA = data; //SET DATA
	EECON2 = 0x55;        
	EECON2 = 0xaa;        
	EECON1bits.WR=1; //WRITE
	while (EECON1bits.WR == 1);
} 

/* LABORIUS SECTION ########################################################################################### */

//////////////////////////////////////////////////////////////////////
//   can_send_laborius_packet
//////////////////////////////////////////////////////////////////////
//
//   Description: Fills a TX buffer with a modified message and sends
//                a Request To Send.
//
//   Input: object_id specifying the TX buffer to use
//   Output: NONE
//   Input/Output: message
//   Returned value: NONE
//
//////////////////////////////////////////////////////////////////////
unsigned char can_send_laborius_packet(LABORIUS_MESSAGE *message)
{
   unsigned char TXBxSIDH = 0;
   unsigned char TXBxSIDL = 0;
   unsigned char TXBxEIDH = 0;
   unsigned char TXBxEIDL = 0;
   
	if(MSG_SILENCE == 0){
	   if (message) {
	
		   // TXBxSIDH: priority + type
		   TXBxSIDH = (message->msg_priority << 5) | (message->msg_type >>3);
		
		   // TXBxSIDL: extended frame + type + eeprom_ram + read_write
		   TXBxSIDL = (message->msg_type << 5) | 0x08;
		   TXBxSIDL |= ((message->msg_eeprom_ram & 0x01) << 1) | (message->msg_read_write & 0x01);
		
		   // TXBxEID8: command
		   TXBxEIDH = (message->msg_cmd);
		
		   // TXBxEID0: destination
		   TXBxEIDL = (message->msg_dest);
		
		   //find emtpy transmitter
		   //map access bank addresses to empty transmitter
		   if (!TXB0CONbits.TXREQ) {
		      
		      //priority (highest priority)
			  TXB0CONbits.TXPRI0 = 1; 
			  TXB0CONbits.TXPRI1 = 1;
			  
			  //datalength + remote request
			  TXB0DLC = (message->msg_data_length | (message->msg_remote << 6));
			        
			  //set CAN ID
			  TXB0SIDH = TXBxSIDH;
			  TXB0SIDL = TXBxSIDL;
			  TXB0EIDH = TXBxEIDH;
			  TXB0EIDL = TXBxEIDL;
			   
		      //copy 8 CAN data bytes
			  TXB0D0 = message->msg_data[0]; 
			  TXB0D1 = message->msg_data[1]; 
			  TXB0D2 = message->msg_data[2]; 
			  TXB0D3 = message->msg_data[3]; 
			  TXB0D4 = message->msg_data[4]; 
			  TXB0D5 = message->msg_data[5]; 
			  TXB0D6 = message->msg_data[6]; 
			  TXB0D7 = message->msg_data[7]; 
		
		   	  //enable transmission
		      TXB0CONbits.TXREQ=1;
		   }
		   else if (!TXB1CONbits.TXREQ) {
		      //CANCONbits.WIN0 = 1;
			  //CANCONbits.WIN1 = 1;
		      //CANCONbits.WIN2 = 0;
		
		      //priority (highest priority)
			  TXB1CONbits.TXPRI0 = 1; 
			  TXB1CONbits.TXPRI1 = 1;
			  
			  //datalength + remote request
			  TXB1DLC = (message->msg_data_length | (message->msg_remote << 6));
			        
			  //set CAN ID
			  TXB1SIDH = TXBxSIDH;
			  TXB1SIDL = TXBxSIDL;
			  TXB1EIDH = TXBxEIDH;
			  TXB1EIDL = TXBxEIDL;
			   
		      //copy 8 CAN data bytes
			  TXB1D0 = message->msg_data[0]; 
			  TXB1D1 = message->msg_data[1]; 
			  TXB1D2 = message->msg_data[2]; 
			  TXB1D3 = message->msg_data[3]; 
			  TXB1D4 = message->msg_data[4]; 
			  TXB1D5 = message->msg_data[5]; 
			  TXB1D6 = message->msg_data[6]; 
			  TXB1D7 = message->msg_data[7]; 
		
			  //enable transmission
		      TXB1CONbits.TXREQ=1;
		   }
		   else if (!TXB2CONbits.TXREQ) {
		      //CANCONbits.WIN0 = 0;
			  //CANCONbits.WIN1 = 1;
		      //CANCONbits.WIN2 = 0;
		
		      //priority (highest priority)
			  TXB2CONbits.TXPRI0 = 1; 
			  TXB2CONbits.TXPRI1 = 1;
			  
			  //datalength + remote request
			  TXB2DLC = (message->msg_data_length | (message->msg_remote << 6));
			        
			  //set CAN ID
			  TXB2SIDH = TXBxSIDH;
			  TXB2SIDL = TXBxSIDL;
			  TXB2EIDH = TXBxEIDH;
			  TXB2EIDL = TXBxEIDL;
			   
		      //copy 8 CAN data bytes
			  TXB2D0 = message->msg_data[0]; 
			  TXB2D1 = message->msg_data[1]; 
			  TXB2D2 = message->msg_data[2]; 
			  TXB2D3 = message->msg_data[3]; 
			  TXB2D4 = message->msg_data[4]; 
			  TXB2D5 = message->msg_data[5]; 
			  TXB2D6 = message->msg_data[6]; 
			  TXB2D7 = message->msg_data[7]; 
		
		 	   //enable transmission
		       TXB2CONbits.TXREQ=1;
			}
			else {
				return(1);
			}
	   
	   		return 0;
	   	}
	   	else {
			return 1;
		}//End if/else message
	}
	else{
		//We dont want to stay for ever in a while(1)
		return 0;
	}//End if/else MSG_SILENCE
}

//////////////////////////////////////////////////////////////////////
//   can_recv_laborius_packet
//////////////////////////////////////////////////////////////////////
//
//   Description: Extract RX buffer message and put it in a message
//                structure.
//
//   Input: object_id specifying the TX buffer to use
//   Output: NONE
//   Input/Output: message
//   Returned value: NONE
//
//////////////////////////////////////////////////////////////////////
unsigned char can_recv_laborius_packet(LABORIUS_MESSAGE *message)
{
	unsigned char retval = 0;

	if (message) {

	    if (RXB0CONbits.RXFUL) {
        
			//clear interrupt flag
			PIR3bits.RXB0IF = 0;
               
			//clear overflow bit
			message->msg_overflow = COMSTATbits.RX1OVFL;
        	COMSTATbits.RX1OVFL = 0;
			
			//data length
			message->msg_data_length = RXB0DLC & 0x0F;        

			//RTR
			message->msg_remote = RXB0DLC >> 6;
    	
			//RXBxSIDH
      		message->msg_priority = RXB0SIDH >> 5;
      		message->msg_type = RXB0SIDH << 3;

      		// RXBxSIDL
      		message->msg_type |= RXB0SIDL >> 5;
      		message->msg_read_write = RXB0SIDL & 0x01;
            message->msg_eeprom_ram = (RXB0SIDL & 0x02) >> 1;

      		// RXBxEIDH
      		message->msg_cmd  = RXB0EIDH;

      		// RXBxEIDL
      	    message->msg_dest = RXB0EIDL;

		
			//COPY DATA
			message->msg_data[0] = RXB0D0;
			message->msg_data[1] = RXB0D1;
			message->msg_data[2] = RXB0D2;
			message->msg_data[3] = RXB0D3;
			message->msg_data[4] = RXB0D4;
			message->msg_data[5] = RXB0D5;
			message->msg_data[6] = RXB0D6;
			message->msg_data[7] = RXB0D7;
		
			message->msg_filter_hit = RXB0CONbits.FILHIT0;
		
			//done with RXB0
			RXB0CONbits.RXFUL=0;

			retval = 1;
	    }
	    else if ( RXB1CONbits.RXFUL )
	    {   
	
			//clear interrupt flag
			PIR3bits.RXB1IF = 0;
	
			//clear overflow bit
			//message->msg_overflow = COMSTATbits.RX2OVFL;
	       	//COMSTATbits.RX2OVFL = 0;

			message->msg_overflow = COMSTATbits.RX1OVFL;
	       	COMSTATbits.RX1OVFL = 0;
			
			//data length
			message->msg_data_length = RXB1DLC & 0x0F;        
	
			//RTR
			message->msg_remote = RXB1DLC >> 6;
	   	
			//RXBxSIDH
	   		message->msg_priority = RXB1SIDH >> 5;
	   		message->msg_type = RXB1SIDH << 3;
	
	   		// RXBxSIDL
	   		message->msg_type |= RXB1SIDL >> 5;
	   		message->msg_read_write = RXB1SIDL & 0x01;
	        message->msg_eeprom_ram = (RXB1SIDL & 0x02) >> 1;
	
	   		// RXBxEIDH
	   		message->msg_cmd  = RXB1EIDH;
	
	   		// RXBxEIDL
	   	    message->msg_dest = RXB1EIDL;
	
		
			//COPY DATA
			message->msg_data[0] = RXB1D0;
			message->msg_data[1] = RXB1D1;
			message->msg_data[2] = RXB1D2;
			message->msg_data[3] = RXB1D3;
			message->msg_data[4] = RXB1D4;
			message->msg_data[5] = RXB1D5;
			message->msg_data[6] = RXB1D6;
			message->msg_data[7] = RXB1D7;
		    				
			message->msg_filter_hit = RXB1CON & 0b00000111;


			RXB1CONbits.RXFUL=0;

			retval = 1;
	    }
	    else {
	      retval = 0;
	    }
	}
	else {
		retval = 0;
	}
 	return retval;
}

//////////////////////////////////////////////////////////////////////
//   can_apply_accept_mask
//////////////////////////////////////////////////////////////////////
//
//   Description: Sets the MCP2510 in configuration mode
//                Updates the specified mask and configures it to its
//                previous mode
//
//   Input: object_id specifying the mask number
//   Output: NONE
//   Input/Output: mask
//   Returned value: NONE
//
//////////////////////////////////////////////////////////////////////
void can_apply_accept_mask(LABORIUS_MASK *mask, unsigned char mask_id)
{   
   unsigned char TXBxSIDH = 0;
   unsigned char TXBxSIDL = 0;
   unsigned char TXBxEIDH = 0;
   unsigned char TXBxEIDL = 0;
    
   if (mask) {

	   // TXBxSIDH: priority + type
	   TXBxSIDH = (mask->mask_priority << 5) | (mask->mask_type >>3);
	
	   // TXBxSIDL: extended frame + type
	   TXBxSIDL = (mask->mask_type << 5);
	
	   // TXBxEID8: command
	   TXBxEIDH = (mask->mask_cmd);
	
	   // TXBxEID0: destination
	   TXBxEIDL = (mask->mask_dest);
	
	   switch(mask_id) {
	      case 0:
			RXM0SIDH = TXBxSIDH;
			RXM0SIDL = TXBxSIDL;
			RXM0EIDH = TXBxEIDH;
			RXM0EIDL = TXBxEIDL;
	      break;
	
	      case 1:
			RXM1SIDH = TXBxSIDH;
			RXM1SIDL = TXBxSIDL;
			RXM1EIDH = TXBxEIDH;
			RXM1EIDL = TXBxEIDL;
	      break;
	   }
   }
}

//////////////////////////////////////////////////////////////////////
//   can_apply_filter
//////////////////////////////////////////////////////////////////////
//
//   Description: Sets the MCP2510 in configuration mode
//                Updates the specified filter and configures it to its
//                previous mode
//
//   Input: object_id specifying the filter number
//   Output: NONE
//   Input/Output: filter
//   Returned value: NONE
//
//////////////////////////////////////////////////////////////////////
void can_apply_filter(LABORIUS_FILTER *filter, unsigned char filter_id)
	{   
	   unsigned char TXBxSIDH = 0;
	   unsigned char TXBxSIDL = 0;
	   unsigned char TXBxEIDH = 0;
	   unsigned char TXBxEIDL = 0;
	   
	   if (filter) {
		   // TXBxSIDH: priority + type
		   TXBxSIDH = (filter->filter_priority << 5) | (filter->filter_type >>3);   

		   // TXBxSIDL: extended frame + type
		   TXBxSIDL = (filter->filter_type << 5) | 0x08;
		
		   // TXBxEID8: command
		   TXBxEIDH = (filter->filter_cmd);
		
		   // TXBxEID0: destination
		   TXBxEIDL = (filter->filter_dest);
		
		   switch(filter_id) {
		      case 0:
				RXF0SIDH = TXBxSIDH;
				RXF0SIDL = TXBxSIDL;
				RXF0EIDH = TXBxEIDH;
				RXF0EIDL = TXBxEIDL;
		      break;
		
		      case 1:
				RXF1SIDH = TXBxSIDH;
				RXF1SIDL = TXBxSIDL;
				RXF1EIDH = TXBxEIDH;
				RXF1EIDL = TXBxEIDL;
		      break;
		
		      case 2:
				RXF2SIDH = TXBxSIDH;
				RXF2SIDL = TXBxSIDL;
				RXF2EIDH = TXBxEIDH;
				RXF2EIDL = TXBxEIDL;
		      break;
		
		      case 3:
				RXF3SIDH = TXBxSIDH;
				RXF3SIDL = TXBxSIDL;
				RXF3EIDH = TXBxEIDH;
				RXF3EIDL = TXBxEIDL;
		      break;
		
		      case 4:
				RXF4SIDH = TXBxSIDH;
				RXF4SIDL = TXBxSIDL;
				RXF4EIDH = TXBxEIDH;
				RXF4EIDL = TXBxEIDL;
		      break;
		
		      case 5:
				RXF5SIDH = TXBxSIDH;
				RXF5SIDL = TXBxSIDL;
				RXF5EIDH = TXBxEIDH;
				RXF5EIDL = TXBxEIDL;
		      break;
		   }
	}
}

//////////////////////////////////////////////////////////////////////
//   can_transceiver()
//////////////////////////////////////////////////////////////////////
//
//   Description: MCP2510 unsigned charerrupt sub-routine
//                Handles all MCP2510 unsigned charerruptions until none are left
//
//   Input: NONE
//   Output: NONE
//   Input/Output: NONE
//   Returned value: NONE
//
//////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////
void can_transceiver(unsigned char can_addr)
{
   LABORIUS_MESSAGE msg_value;      
   unsigned char buffer_size = 0;
   unsigned char call_proc_message = 1;

   while(can_recv_laborius_packet(&g_rMessage)) {
      //Analyse for boot mode and I'm alive signal
      switch(g_rMessage.msg_type){
      case CAN_TYPE_EMERGENCY:

         switch(g_rMessage.msg_cmd){
            case CAN_EMERGENCY_CMD_PROGRAM:
				if(g_rMessage.msg_dest == can_addr){
					INTCONbits.GIEH = 0; //disable high interrupts
					INTCONbits.GIEL = 0; //disable low interrupts
					//indicate to the bootloader that we are gonna program				
					call_proc_message = 0;
	
					can_write_eeprom(0xFF,0xFF);
					
					//wait 10ms
					Delay10KTCYx(10);
									
					//reset!
					Reset();
				}
               break;
         }//end switch msg_cmd

         break;

      case CAN_TYPE_EVENTS:
		//React only to your ID and the General ID but not to other addr.
		if((g_rMessage.msg_dest == can_addr) || (g_rMessage.msg_dest ==0xFF)){
			switch(g_rMessage.msg_cmd){
			   case CAN_EVENTS_CMD_ALIVE:
				    //Send i'm alive
				    can_send_im_alive(can_addr);
					call_proc_message = 0;
				break;
				case CAN_EVENTS_CMD_SILENCE:
					//0 = Driver will send all msg ; 1 = Driver will not send msg except for the IM_ALIVE signal
					MSG_SILENCE = g_rMessage.msg_data[0];
					call_proc_message = 0;
				break;
			}//end switch msg_cmd
		}
		break;

      case CAN_TYPE_REQUEST_DATA:
		if(g_rMessage.msg_dest == can_addr){
	         switch(g_rMessage.msg_read_write){
	            case CAN_REQUEST_READ:
	               //Determine buffer size that will be send
	               buffer_size = ((g_rMessage.msg_cmd & SUB_SECTION_MASK) >> SUB_SECTION_SHIFT);

	               if(buffer_size > BIT_SET_LIMITS){
	                  buffer_size = buffer_size - BIT_SET_LIMITS;
	               }
	               else{
	                  buffer_size = 1;
	               }

	               msg_value.msg_priority = 0;
	               msg_value.msg_type = CAN_TYPE_REQUEST_DATA;
	               msg_value.msg_eeprom_ram = g_rMessage.msg_eeprom_ram;
				   msg_value.msg_read_write = g_rMessage.msg_read_write;
	               msg_value.msg_cmd = g_rMessage.msg_cmd;
	               msg_value.msg_dest = can_addr;
	               msg_value.msg_data_length = buffer_size;
	               read_data_flow_table((g_rMessage.msg_cmd & SECTION_MASK),((g_rMessage.msg_cmd & SUB_SECTION_MASK) >> SUB_SECTION_SHIFT),g_rMessage.msg_eeprom_ram,msg_value.msg_data);
	               msg_value.msg_remote = 0;
	               while(can_send_laborius_packet(&msg_value)){;}
				   call_proc_message = 0;
	               break;
	
	            case CAN_REQUEST_WRITE:
	               // We don't want to overwrite EEPROM table information
	               if(!((g_rMessage.msg_cmd & SECTION_MASK) == 0x00 && (g_rMessage.msg_eeprom_ram == CAN_REQUEST_EEPROM))){
	                  write_data_flow_table((g_rMessage.msg_cmd & SECTION_MASK),((g_rMessage.msg_cmd & SUB_SECTION_MASK) >> SUB_SECTION_SHIFT) ,g_rMessage.msg_eeprom_ram,g_rMessage.msg_data);
	               }
				   call_proc_message = 0;
	               break;
	
	         }//end sub-switch read_write
		}//End if can_addr
		break;  
      }

      //processe the received message
	  if (call_proc_message) {
      	proc_message(&g_rMessage);
	  }
   }//while
}



//////////////////////////////////////////////////////////////////////
//   can_send_im_alive
//////////////////////////////////////////////////////////////////////
//
//   Description: Send a I'M Alive can message
//
//   Input: can_addr (Address read in the EEPROM at 0XFE)
//   Output: NONE
//   Input/Output: NONE
//   Returned value: NONE
//
//////////////////////////////////////////////////////////////////////
void can_send_im_alive(unsigned char can_addr)
{
	LABORIUS_MESSAGE msg;
	unsigned char params_buffer[8];
	unsigned char temp_silence = 0;
	
	//Remember the silence mode
	temp_silence = MSG_SILENCE;

	msg.msg_priority = 0x00;
	msg.msg_type = CAN_TYPE_EVENTS;
	msg.msg_cmd = CAN_EVENTS_CMD_ALIVE;
	msg.msg_dest = can_addr;
	msg.msg_eeprom_ram = CAN_REQUEST_EEPROM;
	msg.msg_read_write = CAN_REQUEST_READ;
	msg.msg_data_length = 2;
	//NEW ADDON DONE BY PL 17 June 2005
	read_data_flow_table(0x00,READ_WRITE_6_BYTE,CAN_REQUEST_EEPROM,params_buffer);
	msg.msg_data[0] = params_buffer[5]; //Mode
	msg.msg_data[1] = params_buffer[1]; //Project ID
	msg.msg_remote = 0;

	//Activate all msg to be sent
	MSG_SILENCE = 0;
	while(can_send_laborius_packet(&msg));
	//Fetch back the last known silence mode
	MSG_SILENCE = temp_silence;
	
}

/* DATA FLOW TABLE FUNCTIONS ################################################################### */

//////////////////////////////////////////////////////////////////////
//   read_data_flow_table
//////////////////////////////////////////////////////////////////////
//
//   Description: read 8 byte in memory
//
//   Input: section_id (CAN_REQUEST_DATA_FLOW_00 to CAN_REQUEST_DATA_FLOW_17), buffer[8]
//   Output: read succesfull/not succesfull
//
//   DESCRIPTION :   This function read 8 bytes in memory based on "section_id" and
//                   fill buffer[].  CAN_REQUEST_DATA_FLOW_00 and CAN_REQUEST_DATA_FLOW_01
//                   refered to EEPROM memory location.  Every others addr refered to
//                   RAM memory.  If the reading is successfull "1" is returned else "0"
//////////////////////////////////////////////////////////////////////

unsigned char read_data_flow_table(unsigned char section_id,
	unsigned char sub_section_id,
	unsigned char mem_type, 
	unsigned char *buffer)
{
   unsigned char i = 0;
   unsigned char base_addr = 0;
   unsigned char success = 1;

   if (section_id < NUM_SECTION) { 
	   switch(mem_type){
	      case CAN_REQUEST_RAM:
	
	         //Sequentiel read in memory base on base_addr
	         if(sub_section_id > BIT_SET_LIMITS){
	            //Ram section address computation
	            base_addr = section_id * DATA_FLOW_ENTRY;
	
	            //Looking if we won't bust ram allocated memory
	            if((base_addr + (sub_section_id - BIT_SET_LIMITS)) < MAX_RAM_DATA_ENTRY){
	               //Looking if we won't read more than 8 bytes in a row
	               if((sub_section_id - BIT_SET_LIMITS) <= DATA_FLOW_ENTRY){
	                  //Filling read buffer
	                  for(i=0;i<(sub_section_id - BIT_SET_LIMITS);i++){
	                     buffer[i] = DATA_FLOW_TABLE[base_addr + i];
	                  }
	               }
	               else{
	                  //Trying to read more than 8 byte in a row
	                  success = 0;
	               }
	            }
	            else{
	               //Trying to read outside Ram memory
	               success = 0;
	            }
	         }
	         //Random byte access memory 
	         else{
	            //Ram specific address computation
	            base_addr = (section_id * DATA_FLOW_ENTRY) + sub_section_id;
	
	            //Looking if we won't bust ram allocated memory
	            if(base_addr < MAX_RAM_DATA_ENTRY){
	               buffer[0] = DATA_FLOW_TABLE[base_addr];
	            }
	            else{
	               //Trying to read outside Ram memory
	               success = 0;
	            }
	         }
	         break;
	
	      case CAN_REQUEST_EEPROM:
	         //Sequentiel write in memory base on base_addr
	         if(sub_section_id > BIT_SET_LIMITS){
	            //Eeprom section address computation
	            base_addr = section_id * DATA_FLOW_ENTRY;
	
	            //Looking if we won't bust EEPROM memory
	            if((base_addr + (sub_section_id - BIT_SET_LIMITS)) < MAX_EEPROM_DATA_ENTRY){
	               //Looking if we won't read more than 8 bytes in a row
	               if((sub_section_id - BIT_SET_LIMITS) <= DATA_FLOW_ENTRY){
	                  //Filling read buffer
	                  for(i=0;i<(sub_section_id - BIT_SET_LIMITS);i++){
							buffer[i] = can_read_eeprom(base_addr + i );
	                  }
	               }
	               else{
	                  //Trying to write more than 8 byte in a row
	                  success = 0;
	               }
	            }
	            else{
	               //Trying to write outside EEPROM memory
	               success = 0;
	            }
	         }
	         //EEPROM byte access memory
	         else{
	            //EEPROM specific address computation
	            base_addr = (section_id * DATA_FLOW_ENTRY) + sub_section_id;
	
	            //Looking if we won't bust EEPROM allocated memory
	            if(base_addr < MAX_EEPROM_DATA_ENTRY){              				
					buffer[0] = can_read_eeprom(base_addr);
	            }
	            else{
	               //Trying to read outside EEPROM memory
	               success = 0;
	            }
	         }
	         //Successfull reading was done on EEPROM MEM
	         if(success){
	            EEPROM_RD = 1;
	            EEPROM_SECTION_ID = section_id;
	         }
	
	         break;
	   }
 	}

   return success;
}
//////////////////////////////////////////////////////////////////////
//   write_data_flow_table
//////////////////////////////////////////////////////////////////////
//
//   Description: writes 8 byte in memory
//
//   Input: section_id (CAN_REQUEST_DATA_FLOW_00 to CAN_REQUEST_DATA_FLOW_17), buffer[8]
//   Output: write succesfull/not succesfull
//
//   DESCRIPTION :   This function writes 8 bytes in memory based on buffer[] and
//                   "section_id".  CAN_REQUEST_DATA_FLOW_00 and CAN_REQUEST_DATA_FLOW_01
//                   refered to EEPROM memory location.  Every others addr refered to
//                   RAM memory.  If the writing is successfull "1" is returned else "0"
//////////////////////////////////////////////////////////////////////
unsigned char write_data_flow_table(unsigned char section_id,unsigned char sub_section_id,unsigned char mem_type, unsigned char *buffer)
{
   	unsigned char i = 0;
   	unsigned char base_addr = 0;
   	unsigned char success = 1;
	unsigned char interrupt_detected = 0;

   	switch(mem_type){
      case CAN_REQUEST_RAM:

         //Sequentiel write in memory base on base_addr
         if(sub_section_id > BIT_SET_LIMITS){
            //Ram section address computation
            base_addr = section_id * DATA_FLOW_ENTRY;

            //Looking if we won't bust ram allocated memory
            if((base_addr + (sub_section_id - BIT_SET_LIMITS)) <= MAX_RAM_DATA_ENTRY){
               //Looking if we won't read more than 8 bytes in a row
               if((sub_section_id - BIT_SET_LIMITS) <= DATA_FLOW_ENTRY){
                  //Filling write buffer
                  for(i=0;i<(sub_section_id - BIT_SET_LIMITS);i++){
                     DATA_FLOW_TABLE[base_addr + i] = buffer[i];
                  }
               }
               else{
                  //Trying to read more than 8 byte in a row
                  success = 0;
               }
            }
            else{
               //Trying to read outside Ram memory
               success = 0;
            }
         }
         //Random byte access memory
         else{
            //Ram specific address computation
            base_addr = (section_id * DATA_FLOW_ENTRY) + sub_section_id;

            //Looking if we won't bust ram allocated memory
            if(base_addr <= MAX_RAM_DATA_ENTRY){
               DATA_FLOW_TABLE[base_addr] = buffer[0];
            }
            else{
               //Trying to read outside Ram memory
               success = 0;
            }
         }
         break;

      case CAN_REQUEST_EEPROM:
         //Sequentiel write in memory base on base_addr
         if(sub_section_id > BIT_SET_LIMITS){
            //Eeprom section address computation
            base_addr = section_id * DATA_FLOW_ENTRY;

            //Looking if we won't bust EEPROM allocated memory
            if((base_addr + (sub_section_id - BIT_SET_LIMITS)) <= MAX_EEPROM_DATA_ENTRY){
               //Looking if we won't read more than 8 bytes in a row
               if((sub_section_id - BIT_SET_LIMITS) <= DATA_FLOW_ENTRY){
                  //Filling write buffer
                  for(i=0;i<(sub_section_id - BIT_SET_LIMITS);i++){
			if(INTCONbits.GIEH == 1 && INTCONbits.GIEL == 1){
				INTCONbits.GIEH = 0;
				INTCONbits.GIEL = 0;
				interrupt_detected = 1;
			}
                     can_write_eeprom((base_addr+i),buffer[i]);
			if(interrupt_detected == 1){
				INTCONbits.GIEH = 1;
				INTCONbits.GIEL = 1;
				interrupt_detected = 0;
			}
                  }
               }
               else{
                  //Trying to read more than 8 byte in a row
                  success = 0;
               }
            }
            else{
               //Trying to read outside EEPROM memory
               success = 0;
            }
         }
         //EEPROM bit access memory
         else{
            //EEPROM specific address computation
            base_addr = (section_id * DATA_FLOW_ENTRY) + sub_section_id;

            //Looking if we won't bust EEPROM allocated memory
            if(base_addr <= MAX_EEPROM_DATA_ENTRY){
			if(INTCONbits.GIEH == 1 && INTCONbits.GIEL == 1){
				INTCONbits.GIEH = 0;
				INTCONbits.GIEL = 0;
				interrupt_detected = 1;
			}
              	can_write_eeprom(base_addr,buffer[0]);
			if(interrupt_detected == 1){
				INTCONbits.GIEH = 1;
				INTCONbits.GIEL = 1;
				interrupt_detected = 0;
			}
            }
            else{
               //Trying to read outside EEPROM memory
               success = 0;
            }
         }

         //Successfull writing was done on EEPROM MEM
         if(success){
            EEPROM_WR = 1;
            EEPROM_SECTION_ID = section_id;
         }

         break;
   }

   return success;
}

unsigned char eeprom_hit(void)
{
   unsigned char retval = 0;

   retval = EEPROM_WR;
   EEPROM_WR = 0;

   return retval;
}


/* INTERNAL DRIVER SECTION ##################################################################### */

////////////////////////////////////////////////////////////////////////
//
// can_init()
//
// Initializes PIC18xxx8 CAN peripheral.  Sets the RX filter and masks so the
// CAN peripheral will receive all incoming IDs.  Configures both RX buffers
// to only accept valid valid messages (as opposed to all messages, or all
// extended message, or all standard messages).  Also sets the tri-state
// setting of B2 to output, and B3 to input (apparently the CAN peripheral
// doesn't keep track of this)
//
// The constants (CAN_USE_RX_DOUBLE_BUFFER, CAN_ENABLE_DRIVE_HIGH,
// CAN_ENABLE_CAN_CAPTURE) are given a default define in the can-18xxx8.h file.
// These default values can be overwritten in the main code, but most
// applications will be fine with these defaults.
//
//////////////////////////////////////////////////////////////////////////////
void can_init (LABORIUS_FILTER *filter, LABORIUS_MASK *mask) {
   unsigned char i = 0;

   //PIN B3 (RX) is in, B2 (TX) is out
   TRISBbits.TRISB3 = 1;
   TRISBbits.TRISB2 = 0;

   //Get in config mode for now
   can_set_mode(CAN_OP_CONFIG);

   //make sure to possess a filter with dest = 0xFF
   filter[5].filter_dest |= 0xFF;

   //Set Filter
   for(i=0;i<6;i++){
      can_apply_filter(&filter[i],i);
   }

   //make sure that mask[1].dest = at least 0xFF
   mask[1].mask_dest |= 0xFF;

   //Set MASK
   for(i=0;i<2;i++){
      can_apply_accept_mask(&mask[i],i);
   }

   //Set Bit rate
   can_set_baud();

   //Set IO
   RXB0CON=0;

   RXB0CONbits.RXM0=0; //RECIVE VALID EXTENDED MESSAGES
   //RXB0CONbits.RXM1=1; 23 aout 2005 PL
	RXB0CONbits.RXM1=0;

   RXB0CONbits.RXB0DBEN = 1; //RECEIVE BUFFER 0 WILL OVERFLOW TO RECEIVE BUFFER 1


   RXB1CON=RXB0CON; //SAME CONFIGURATION FOR RX BUFFER 1 (EXCEPT DB OVERFLOW)

   //CIOCONbits.ENDRHI=CAN_ENABLE_DRIVE_HIGH;
   //CIOCONbits.CANCAP=CAN_ENABLE_CAN_CAPTURE;
   CIOCON = 0b00100000; //CAN IO control VDD when recessive, no capture

   //ACCESS BANK AREA ALWAYS ON RXB0 INTERRUPTS
	CANCONbits.WIN0=0;
	CANCONbits.WIN1=1;
    CANCONbits.WIN2=1;
   //Get to normal again
   can_set_mode(CAN_OP_NORMAL);  
}

////////////////////////////////////////////////////////////////////////
//
// can_set_baud()
//
// Configures the baud rate control registers.  All the defines here
// are defaulted in the can-18xxx8.h file.  These defaults can, and
// probably should, be overwritten in the main code.
//
// Current defaults are set to work with Microchip's MCP250xxx CAN
// Developers Kit if this PIC is running at 20Mhz.
//
////////////////////////////////////////////////////////////////////////
void can_set_baud(void) {

	//1MBPS
	BRGCON1 = 0b00000000;
	BRGCON2 = 0b10111010;
	BRGCON3 = 0b00000111;
/*
   BRGCON1bits.BRP0=CAN_BRG_PRESCALAR;
   BRGCON1bits.BRP1=CAN_BRG_PRESCALAR >> 1;
   BRGCON1bits.BRP2=CAN_BRG_PRESCALAR >> 2;

   BRGCON1bits.SJW0=CAN_BRG_SYNCH_JUMP_WIDTH;
   BRGCON1bits.SJW1=CAN_BRG_SYNCH_JUMP_WIDTH >> 1;

   BRGCON2bits.PRSEG0=CAN_BRG_PROPAGATION_TIME;
   BRGCON2bits.PRSEG1=CAN_BRG_PROPAGATION_TIME >> 1;

   BRGCON2bits.SEG1PH0=CAN_BRG_PHASE_SEGMENT_1;
   BRGCON2bits.SEG1PH1=CAN_BRG_PHASE_SEGMENT_1 >> 1;

   BRGCON2bits.SAM=CAN_BRG_SAM;
   BRGCON2bits.SEG2PHTS=CAN_BRG_SEG_2_PHASE_TS;


   BRGCON3bits.SEG2PH0=CAN_BRG_PHASE_SEGMENT_2;
   BRGCON3bits.SEG2PH1=CAN_BRG_PHASE_SEGMENT_2 >> 1;
   BRGCON3bits.SEG2PH2=CAN_BRG_PHASE_SEGMENT_2 >> 2;

   BRGCON3bits.WAKFIL=CAN_BRG_WAKE_FILTER;
*/
}

void can_set_mode(unsigned char mode) {

   CANCONbits.REQOP0 = mode;
   CANCONbits.REQOP1 = mode >> 1;
   CANCONbits.REQOP2 = mode >> 2;
	   
   //CIOCON = 0b00100000; //CAN IO control
   while((CANSTAT >> 5) != mode );
}








