//////////////////////////////////////////////////////////////////////
//   main.c
//////////////////////////////////////////////////////////////////////
//
//   Description: Acquire serial commands at 57 600 bauds.  Send CAN 
//                commands to the servo controller
//
//   Name: Pierre Lepage
//   Date de cration : 2006/10/06
//
//   Modification:
//
//   Name                  	Date			Description
//   Pierre Lepage	    	2006-12-05	Whisker are now pooled
//   Pierre Lepage		2007-04-20	Modification of servo 8 (neck yaw) its now a 180 deg
//   Pierre Lepage		2007-04-20	Adjustment of some of the soft limit switch
//   Pierre Lepage		2007-04-20	Modification of the function init_face() (Eyes yaw section)
//   Pierre Lepage		2007-11-19   Norm 270 added in main.c & main.h
//   Pierre Lepage		2007-11-19	command "mra" & "mla" now use norm_270 instead of norm_180
//   Pierre Lepage		2007-11-29   Function addon "norm_servor()" see comment for more information
//   Pierre Lepage		2007-11-29   Modification of "init_face()"
//   Pierre Lepage		2007-11-29	Bug fixed in "analogue_detection()"
//   Pierre Lepage		2007-11-29	Modification of function "limit_position()"
//   Pierre Lepage		2007-11-29	Modification of function "do_action()" => now use "norm_servo()"
//
//////////////////////////////////////////////////////////////////////
//               Copyright by RoboMotio inc.  2006
//////////////////////////////////////////////////////////////////////

#include <p18F458.h>
#include <timers.h>
#include <delays.h>
#include <portb.h>
#include <stdlib.h>
#include <ctype.h>
#include "main.h"
#include "can18_driver.h"
#include "can_shared.h"
#include "serial_driver.h"
#include "analogique.h"

//Global message to send via CAN
LABORIUS_MESSAGE message_to_send;
LABORIUS_MESSAGE message_debug;
// Control Structure variable --------------------------------------------------
CTRL_PARAMS ctrl_panel;
RCV_STRING ctrl_string;

unsigned char Can_Addr = 0x00;
unsigned char DEBUG = 0;

void highpriority_isr(void);
void lowpriority_isr(void);

/* High priority interrupt vector */
#pragma code high_vector=0x008
void high_interrupt (void)
{
_asm GOTO highpriority_isr _endasm
}

/* Low priority interrupt vector */
#pragma code low_vector=0x018
void low_interrupt (void)
{
_asm GOTO lowpriority_isr _endasm
}

/***********************/
/* ISR implementations */
/***********************/
/* High priority interrupt */
//See section 2.9.2.4 ISR CONTEXT SAVING of C18 compiler user guide
//#pragma interrupt highpriority_isr save=PROD , section(".tmpdata"), TBLPTR, TABLAT
#pragma interrupt highpriority_isr //save=section(".tmpdata"),section("MATH_DATA"),PROD
void highpriority_isr(void)
{
	unsigned char value = 0;

	//PIR1 =>Serial RX RCIF
	if(PIR1bits.RCIF){
		//When RCREG is read it will automatically clear the RCIF flag
		value = RCREG;
		
		*RS232_RX_Over = RCSTAbits.OERR;						//Probing Overflow Error
		*RS232_RX_Err = RCSTAbits.FERR;							//Probing RX Error

		if((RCSTAbits.OERR==1)){
			RCSTAbits.CREN = 0;
			RCSTAbits.CREN = 1;
		}

		if((RCSTAbits.FERR==1)){
			value = RCREG;
		}
	
		input_parser(value);
	}

}

/* Low priority interrupt */
//See section 2.9.2.4 ISR CONTEXT SAVING of C18 compiler user guide
//#pragma interruptlow lowpriority_isr save=section(".tmpdata"), PROD
#pragma interruptlow lowpriority_isr //save=section(".tmpdata"),section("MATH_DATA"),PROD, TBLPTR, TABLAT
void lowpriority_isr(void)
{
	//INTCON => Timer0, RbIF
	if(INTCONbits.TMR0IF){
		INTCONbits.TMR0IF = 0;
// ---------------------------------------------------------
		//Modification 5 Dec 2006 
		pooling_whisker_detection();
// ---------------------------------------------------------
		led();
	}

//----------------------------------------------------------------------------------------
	//Modification 5 Dec 2006
	//Interrupt on Direction (RB4-RB7)
	//if(INTCONbits.RBIF){
		//clear interrupt flag
	//	INTCONbits.RBIF = 0;
	//	whisker_detection();
	//}
//----------------------------------------------------------------------------------------

}
//----------------------------------------- INTERRUPT FUNCTION --------------------------------------------
//////////////////////////////////////////////////////////////////////
//   LED
//////////////////////////////////////////////////////////////////////
//
//   Description: Blink the led on RA4
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void led(void)
{
	//clear interrupt flag
	PORTAbits.RA4 = ~(PORTAbits.RA4);
}

//////////////////////////////////////////////////////////////////////
//   WHISKER_DETECTION
//////////////////////////////////////////////////////////////////////
//
//   Description: Not use
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void whisker_detection(void)
{
	if(PORTBbits.RB4 == 0){
		ctrl_panel.whisker_state |= 0x01;
	}

	if(PORTBbits.RB5 == 0){
		ctrl_panel.whisker_state |= 0x02;
	}

	if(PORTBbits.RB6 == 0){
		ctrl_panel.whisker_state |= 0x04;
	}

	if(PORTBbits.RB7 == 0){
		ctrl_panel.whisker_state |= 0x08;
	}
	ctrl_panel.whisker_requested = 1;
}

//////////////////////////////////////////////////////////////////////
//   POOLING WHISKER DETECTION
//////////////////////////////////////////////////////////////////////
//
//   Description: This function is called in timer 0 and check the state
//				  of pins RB4 to RB7.
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void pooling_whisker_detection(void)
{

	unsigned char temp_RB4_value = 0;
	unsigned char temp_RB5_value = 0;
	unsigned char temp_RB6_value = 0;
	unsigned char temp_RB7_value = 0;

	//Since we are pooling, we must be sure that input value are well recorded in the process
	temp_RB4_value = PORTBbits.RB4;
	temp_RB5_value = PORTBbits.RB5;
	temp_RB6_value = PORTBbits.RB6;
	temp_RB7_value = PORTBbits.RB7;

	//Compare old value with the new one
	if(ctrl_panel.old_state_RB4 != temp_RB4_value){
		//Remember last state
		ctrl_panel.old_state_RB4 = temp_RB4_value;
		//Clear the bit
		ctrl_panel.whisker_state &= 0x0E;
		//Fetch the bit at the good place in buffer
		ctrl_panel.whisker_state |= (temp_RB4_value & 0x01);

		ctrl_panel.whisker_requested = 1;
	}

	if(ctrl_panel.old_state_RB5 != temp_RB5_value){
		//Remember last state
		ctrl_panel.old_state_RB5 = temp_RB5_value;
		//Clear the bit
		ctrl_panel.whisker_state &= 0x0D;
		//Fetch the bit at the good place in buffer
		ctrl_panel.whisker_state |= ((temp_RB5_value << 1) & 0x02);

		ctrl_panel.whisker_requested = 1;
	}

	if(ctrl_panel.old_state_RB6 != temp_RB6_value){
		//Remember last state
		ctrl_panel.old_state_RB6 = temp_RB6_value;
		//Clear the bit
		ctrl_panel.whisker_state &= 0x0B;
		//Fetch the bit at the good place in buffer
		ctrl_panel.whisker_state |= ((temp_RB6_value << 2) & 0x04);

		ctrl_panel.whisker_requested = 1;
	}

	if(ctrl_panel.old_state_RB7 != temp_RB7_value){
		//Remember last state
		ctrl_panel.old_state_RB7 = temp_RB7_value;
		//Clear the bit
		ctrl_panel.whisker_state &= 0x07;
		//Fetch the bit at the good place in buffer
		ctrl_panel.whisker_state |= ((temp_RB7_value << 3) & 0x08);

		ctrl_panel.whisker_requested = 1;
	}
	
}


//////////////////////////////////////////////////////////////////////
//   INPUT PARSER
//////////////////////////////////////////////////////////////////////
//
//   Description: This function is called when a byte is detected on the
//				  serial line.  It will parse the data coming from serial.
//				  Serial commands are stored in : ctrl_string structure
//
//   Input: Value (byte received)
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void input_parser(unsigned char value)
{
	unsigned char nb_params = 0;

	//Did we received the str id ?
	if(ctrl_string.str_header == 1){
		if(isalpha(value) != 0){
			//Computing the ID of the cmd
			ctrl_string.str_id += value;

			if(ctrl_string.str_index == 2){
				switch(ctrl_string.str_id){
					case CMD_GBV:
					case CMD_GWS:
					case CMD_GAV:
						//Do the action requested right away
						do_action();
						//Reinitialize the system
						init_rcv_string();
						break;
					default:
						//switch to acquire the params of the function
						ctrl_string.str_header = 0;
						break;
				}
			}
			else{
				ctrl_string.str_index++;
			}
		}
		else{
			//If the user is trying weird thing reinitialise the recv sequence
			if(ctrl_string.str_index != 0){
				init_rcv_string();
			}
		}
	}
	else{
		ctrl_string.str_params_buffer[ctrl_string.str_params_index] = value;

		switch(ctrl_string.str_id){
			case CMD_MRA:
				nb_params = 6;	
				break;
			case CMD_MLA:
				nb_params = 6;
				break;
			case CMD_MRE:
				nb_params = 6;
				break;
			case CMD_MEB:
				nb_params = 4;
				break;
			case CMD_MRM:
				nb_params = 8;
				break;
			case CMD_MRN:
				nb_params = 4;
				break;
			case CMD_MSS:
				nb_params = 3;
				break;
			case CMD_CAN:
				nb_params = 13;
				break;

			default:
				//User try to lurk you !
				init_rcv_string();
				break;
		}

		if(ctrl_string.str_params_index == (nb_params-1)){
			//Do the action requested right away
			do_action();
			//Reinitialize the system
			init_rcv_string();
		}
		else{
			ctrl_string.str_params_index++;
			if(DEBUG == 1){
				*COUNTER = ctrl_string.str_params_index;
			}
		}
	}

}

//////////////////////////////////////////////////////////////////////
//   DO ACTION
//////////////////////////////////////////////////////////////////////
//
//   Description: This function will analyse the commands send over serial
//				  line and send the good CAN command to the servo controller.
//				  It uses the structure : ctrl_string
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void do_action(void)
{
	unsigned int CAN_degree = 0;
	unsigned int value_limited = 0;
	unsigned int i;
	LABORIUS_MESSAGE can_message;

	switch(ctrl_string.str_id){


		case CMD_CAN:
			//Received an explicit CAN command
			//Forwarding to CAN bus...

			//CAN ID
			can_message.msg_priority = ctrl_string.str_params_buffer[0];
			can_message.msg_type = ctrl_string.str_params_buffer[1];
			can_message.msg_remote = (ctrl_string.str_params_buffer[2] >> 4) & 0x01;
			can_message.msg_read_write = (ctrl_string.str_params_buffer[2] >> 5) & 0x01;
			can_message.msg_eeprom_ram = (ctrl_string.str_params_buffer[2] >> 6) & 0x01;
			can_message.msg_data_length = (ctrl_string.str_params_buffer[2] & 0x0F);			
			can_message.msg_cmd = ctrl_string.str_params_buffer[3];
			can_message.msg_dest = ctrl_string.str_params_buffer[4];
	
			//COPY CAN DATA 
			for(i = 0; i < 8; i++)
			{
				can_message.msg_data[i] = ctrl_string.str_params_buffer[5 + i];
			}
			
			//Sending message
			while(can_send_laborius_packet(&can_message));
		break;


		case CMD_MRA:
			//Move right arm
			

			//CAN ID
			can_message.msg_priority = 0;
			can_message.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY ;
			can_message.msg_remote = 0;
			can_message.msg_read_write = 0x01;
			can_message.msg_eeprom_ram = 0x01;
			can_message.msg_data_length = 3;			
			can_message.msg_cmd = 0x93;
			can_message.msg_dest = 0x02; //RIGHT ARM
	
			//Sending message Pitch shoulder
			can_message.msg_data[0] = 0;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[4];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[5];		
			while(can_send_laborius_packet(&can_message));


			//Sending message Yaw shoulder
			can_message.msg_data[0] = 1;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[2];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[3];		
			while(can_send_laborius_packet(&can_message));


			//Sending message Elbow
			can_message.msg_data[0] = 2;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[0];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[1];		
			while(can_send_laborius_packet(&can_message));

			/*
			//Servo 1 - 2 - 3
			//Doing RIGHT ELBOW servo 1	
			message_to_send.msg_data[0] = get_servo_id(1); //servo real id
			value_limited = limit_position(1,ctrl_string.str_params_buffer[0]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_270(value_limited);
			//unsigned char serial_value, unsigned short long multiplicator, unsigned char shift, unsigned int base_tick, unsigned char inversion
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 1031,7,1025,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[1];
			while(can_send_laborius_packet(&message_to_send));

			//Doing RIGHT SHOULDER YAW servo 2
			message_to_send.msg_data[0] = get_servo_id(2); //servo real id
			value_limited = limit_position(2,ctrl_string.str_params_buffer[2]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_270(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 1031,7,1750,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[3];
			while(can_send_laborius_packet(&message_to_send));

			//Doing RIGHT SHOULDER PITCH servo 3
			message_to_send.msg_data[0] = get_servo_id(3); //servo real id
			value_limited = limit_position(3,ctrl_string.str_params_buffer[4]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_270(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 3911,9,250,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[5];
			while(can_send_laborius_packet(&message_to_send));
			*/

			break;
			
		case CMD_MLA:
			//Move left arm

			//CAN ID
			can_message.msg_priority = 0;
			can_message.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY ;
			can_message.msg_remote = 0;
			can_message.msg_read_write = 0x01;
			can_message.msg_eeprom_ram = 0x01;
			can_message.msg_data_length = 3;			
			can_message.msg_cmd = 0x93;
			can_message.msg_dest = 0x01; //LEFT ARM
	
			//Sending message Pitch shoulder
			can_message.msg_data[0] = 0;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[4];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[5];		
			while(can_send_laborius_packet(&can_message));


			//Sending message Yaw shoulder
			can_message.msg_data[0] = 1;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[2];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[3];		
			while(can_send_laborius_packet(&can_message));


			//Sending message Elbow
			can_message.msg_data[0] = 2;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[0];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[1];		
			while(can_send_laborius_packet(&can_message));

			/*
			//Servo 6 - 7 - 4
			//Doing LEFT ELBOW servo 6
			message_to_send.msg_data[0] = get_servo_id(6); //servo real id
			value_limited = limit_position(6,ctrl_string.str_params_buffer[0]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_270(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 15,1,1075,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[1];
			while(can_send_laborius_packet(&message_to_send));

			//Doing LEFT SHOULDER YAW servo 7
			message_to_send.msg_data[0] = get_servo_id(7); //servo real id
			value_limited = limit_position(7,ctrl_string.str_params_buffer[2]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_180(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 1031,7,1750,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[3];
			while(can_send_laborius_packet(&message_to_send));

			//Doing LEFT SHOULDER PITCH servo 4
			message_to_send.msg_data[0] = get_servo_id(4); //servo real id
			value_limited = limit_position(4,ctrl_string.str_params_buffer[4]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_180(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 3911,9,1750,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[5];
			while(can_send_laborius_packet(&message_to_send));
			*/


			break;
			
		case CMD_MRE:
			//Move robot eyes
			//Servo 17 - 16 - 15
			//Doing LEFT EYE YAW servo 16
			message_to_send.msg_data[0] = get_servo_id(16); //servo real id
			value_limited = limit_position(16,ctrl_string.str_params_buffer[0]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-26, 2863,7,525,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[3];
			while(can_send_laborius_packet(&message_to_send));

			//Doing RIGHT EYE YAW servo 17
			message_to_send.msg_data[0] = get_servo_id(17); //servo real id
			value_limited = limit_position(17,ctrl_string.str_params_buffer[2]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-26, 2863,7,625,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[1];
			while(can_send_laborius_packet(&message_to_send));

			//Doing EYE PITCH servo 15
			message_to_send.msg_data[0] = get_servo_id(15); //servo real id
			value_limited = limit_position(15,ctrl_string.str_params_buffer[4]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-25, 89,2,555,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[5];
			while(can_send_laborius_packet(&message_to_send));
			break;
			
		case CMD_MEB:
			//Move Eyebrows
			//Servo 18 - 20
			//Doing EYEBROW LEFT servo 20 
			message_to_send.msg_data[0] = get_servo_id(20); //servo real id
			value_limited = limit_position(20,ctrl_string.str_params_buffer[0]);
			
			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-10, 711,5,223,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[3];
			while(can_send_laborius_packet(&message_to_send));

			//Doing EYEBROW RIGHT servo 18
			message_to_send.msg_data[0] = get_servo_id(18); //servo real id
			value_limited = limit_position(18,ctrl_string.str_params_buffer[2]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-10, 711,5,1778,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[1];
			while(can_send_laborius_packet(&message_to_send));
			
			break;
			
		case CMD_MRM:
			//Move robot mouth
			//Servo 11 - 14 - 13 - 12
			//Doing MOUTH LEFT UPPER servo 13
			message_to_send.msg_data[0] = get_servo_id(13); //servo real id
			value_limited = limit_position(13,ctrl_string.str_params_buffer[0]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,1555,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[5];
			while(can_send_laborius_packet(&message_to_send));

			//Doing MOUTH LEFT LOWER servo 12
			message_to_send.msg_data[0] = get_servo_id(12); //servo real id
			value_limited = limit_position(12,ctrl_string.str_params_buffer[2]);
			
			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,1555,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[7];
			while(can_send_laborius_packet(&message_to_send));

			//Doing MOUTH RIGHT UPPER servo 11
			message_to_send.msg_data[0] = get_servo_id(11); //servo real id
			value_limited = limit_position(11,ctrl_string.str_params_buffer[4]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,445,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[1];
			while(can_send_laborius_packet(&message_to_send));

			//Doing MOUTH RIGHT LOWER servo 14
			message_to_send.msg_data[0] = get_servo_id(14); //servo real id
			value_limited = limit_position(14,ctrl_string.str_params_buffer[6]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,445,0);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[3];
			while(can_send_laborius_packet(&message_to_send));

			break;
		case CMD_MRN:
			//Move robot neck
			//CAN ID
			can_message.msg_priority = 0;
			can_message.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY ;
			can_message.msg_remote = 0;
			can_message.msg_read_write = 0x01;
			can_message.msg_eeprom_ram = 0x01;
			can_message.msg_data_length = 3;			
			can_message.msg_cmd = 0x93;
			
	
			//Sending message Pitch Neck
			can_message.msg_dest = 0x02; 
			can_message.msg_data[0] = 3;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[0];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[1];		
			while(can_send_laborius_packet(&can_message));


			//Sending message Yaw Neck
			can_message.msg_dest = 0x01; 
			can_message.msg_data[0] = 3;
			can_message.msg_data[1] = ctrl_string.str_params_buffer[2];
			can_message.msg_data[2] = ctrl_string.str_params_buffer[3];		
			while(can_send_laborius_packet(&can_message));


			/*
			//NECK PITCH servo 9
			message_to_send.msg_data[0] = get_servo_id(9); //servo real id
			value_limited = limit_position(9,ctrl_string.str_params_buffer[0]);

			//CAN_degree = BASED_CAN_DEGREE - norm_servo_180(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-67, 1391,7,1250,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[1];
			while(can_send_laborius_packet(&message_to_send));

			//NECK YAW servo 8 (modification 6 December 2006 - analog servo)
			message_to_send.msg_data[0] = get_servo_id(8); //servo real id
			value_limited = limit_position(8,ctrl_string.str_params_buffer[2]);

			//MERL Version
			//CAN_degree = BASED_CAN_DEGREE - norm_servo_180(value_limited);
			CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-50, 89,3,1445,1);
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[3];
			while(can_send_laborius_packet(&message_to_send));
			*/
			
			break;
			
		case CMD_MSS:
			//Move single servo
			message_to_send.msg_data[0] = get_servo_id(ctrl_string.str_params_buffer[0]); //servo real id
			value_limited = limit_position(ctrl_string.str_params_buffer[0],ctrl_string.str_params_buffer[1]);

			switch(ctrl_string.str_params_buffer[0]){
				case ELBOW_RIGHT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 1031,7,1025,0);
					break;
				case SHOULDER_YAW_RIGHT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 1031,7,1750,1);
					break;
				case SHOULDER_PITCH_RIGHT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 3911,9,250,0);
					break;
				case ELBOW_LEFT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 15,1,1075,0);
					break;
				case SHOULDER_YAW_LEFT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 1031,7,1750,1);
					break;
				case SHOULDER_PITCH_LEFT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited, 3911,9,1750,1);
					break;
				case NECK_YAW:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-50, 89,3,1445,1);
					break;
				case NECK_PITCH:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-67, 1391,7,1250,1);
					break;
				case MOUTH_UPPER_RIGHT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,445,0);
					break;
				case MOUTH_LOWER_RIGHT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,445,0);
					break;
				case MOUTH_UPPER_LEFT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,1555,1);
					break;
				case MOUTH_LOWER_LEFT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,1555,1);
					break;
				case EYES_PITCH:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-25, 89,2,555,0);
					break;
				case EYE_YAW_LEFT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-26, 2863,7,525,0);
					break;
				case EYE_YAW_RIGHT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-26, 2863,7,625,0);
					break;
				case EYEBROW_RIGHT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-10, 711,5,1778,1);
					break;
				case EYEBROW_LEFT:
					CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-10, 711,5,223,0);
					break;
			}

			/*-----------------------------------------------------
			if(ctrl_string.str_params_buffer[0] < 11){
				if(ctrl_string.str_params_buffer[0] == 8){
					//Modification 6 Dec 2006
					CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
				}
				else{
					CAN_degree = BASED_CAN_DEGREE - norm_servo_180(value_limited);
				}
			}
			else{
				CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
			}
			-------------------------------------------------------*/
			
			message_to_send.msg_data[1] = CAN_degree & 0x00FF;
			message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
			message_to_send.msg_data[3] = ctrl_string.str_params_buffer[2];
			while(can_send_laborius_packet(&message_to_send));

			break;
		case CMD_GBV:
			ctrl_panel.battery_requested = 1;
			break;
		case CMD_GWS:
// ------------------------------------------------------------
			//Modfication 5 Dec 2006
			ctrl_panel.old_state_RB4 = 0xFF;
			ctrl_panel.old_state_RB5 = 0xFF;
			ctrl_panel.old_state_RB6 = 0xFF;
			ctrl_panel.old_state_RB7 = 0xFF;
// ------------------------------------------------------------

			//get whisker sensor
			pooling_whisker_detection();
			break;
		case CMD_GAV:
			ctrl_panel.analogue_requested = 1;
			break;
		default:
			//User try to lurk you !
			init_rcv_string();
			break;
	}
	
}
//---------------------------------------- INTERNAL FUNCTION ----------------------------------------------
//////////////////////////////////////////////////////////////////////
//   proc_message
//////////////////////////////////////////////////////////////////////
//
//   Description: CAN callback from file can18_driver.c. No special action
//				  for this project
// 
//
//   Input: LABORIUS_MESSAGE *message
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void proc_message(LABORIUS_MESSAGE *message) {

	unsigned char i = 0;
	//DIRECT FORWARD TO SERIAL PORT

	char can_recv_buffer[16] = {'c' , 'a' , 'n' , 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	
	can_recv_buffer[3] = message->msg_priority;
	can_recv_buffer[4] = message->msg_type;
	can_recv_buffer[5] = message->msg_data_length;
	can_recv_buffer[5] |= (message->msg_remote << 4);
	can_recv_buffer[5] |= (message->msg_read_write << 5);
	can_recv_buffer[5] |= (message->msg_eeprom_ram << 6);
	can_recv_buffer[6] = message->msg_cmd;
	can_recv_buffer[7] = message->msg_dest;

	for (i = 0; i < message->msg_data_length; i++)
	{
		can_recv_buffer[8 + i] = message->msg_data[i];
	}
	
	//SEND SERIAL...
	send_s_serial(can_recv_buffer,16);
	
}

//////////////////////////////////////////////////////////////////////
//   norm_servo
//////////////////////////////////////////////////////////////////////
//
//   Description:	This function will normalize a value coming from serial
//				line.
//
//   Input: unsigned char serial_value, unsigned short long multiplicator, unsigned char shift, unsigned int base_tick, unsigned char inversion
//   Output: Unsigned char (converted value)
//   Input/Output:
//   Returned value: NONE
//   -----------------------------------------------------------------------
//   SERVO :		1100 us 		= 65285 ticks
//				1900 us 		= 63285 ticks
//				Delta 800 us 	= Delta 2000 ticks
//   -----------------------------------------------------------------------
//   VERSION :	RED0104.01
//				Right Elbow ########################
//				0   deg		= 1100 us + 410 us = 1510 us = 64 260 ticks
//				90 deg		= 1100 us + 700 us = 1800 us = 63 535 ticks
//				Delta 90 deg	= Delta 290 us = Delta 725 ticks (offset value = 0)
//				Multiplicator	= 1031
//				Shift			= 7
//				Base_tick	= 1025 (410us * 2000 / 800us)
//				Inversion		= 0
//
//				Right Shoulder Yaw ####################
//				0   deg		= 1100 us + 700 us = 1800 us = 63 535 ticks
//				90 deg		= 1100 us + 410 us = 1510 us = 64 260 ticks
//				Delta 90 deg	= Delta 290 us = Delta 725 ticks (offset value = 0)
//				Multiplicator	= 1031
//				Shift			= 7
//				Base_tick	= 1750 (700us * 2000 / 800us)
//				Inversion		= 1
//
//				Right Shoulder Pitch ####################
//				0   	deg		= 1100 us + 100 us = 1200 us = 65 035 ticks
//				180 deg		= 1100 us + 650 us = 1750 us = 63 660 ticks
//				Delta 180 deg	= Delta 550 us = Delta 1375 ticks (offset value = 0)
//				Multiplicator	= 3911
//				Shift			= 9
//				Base_tick	= 250 (100us * 2000 / 800us)
//				Inversion		= 0
//
//				Left Elbow #########################
//				0   deg		= 1100 us + 430 us = 1530 us = 64 210 ticks
//				90 deg		= 1100 us + 700 us = 1800 us = 63 535 ticks
//				Delta 90 deg	= Delta 270 us = Delta 675 ticks (offset value = 0)
//				Multiplicator	= 15
//				Shift			= 1
//				Base_tick	= 1075 (430us * 2000 / 800us)
//				Inversion		= 0
//
//				Left Shoulder Yaw ####################
//				0   deg		= 1100 us + 700 us = 1800 us = 63 535 ticks
//				90 deg		= 1100 us + 410 us = 1510 us = 64 260 ticks
//				Delta 90 deg	= Delta 290 us = Delta 725 ticks (offset value = 0)
//				Multiplicator	= 1031
//				Shift			= 7
//				Base_tick	= 1750 (700us * 2000 / 800us)
//				Inversion		= 1
//
//				Left Shoulder Pitch ####################
//				0   	deg		= 1100 us + 700 us = 1800 us = 63 535 ticks
//				180 deg		= 1100 us + 150 us = 1250 us = 64 910 ticks
//				Delta 180 deg	= Delta 550 us = Delta 1375 ticks (offset value = 0)
//				Multiplicator	= 3911
//				Shift			= 9
//				Base_tick	= 1750 (700us * 2000 / 800us)
//				Inversion		= 1
//
//				Neck Pitch #########################
//				67   	deg		= 1100us + 500us = 1600us = 64 035 ticks
//				113 deg		= 1100us + 300us = 1400us = 64 535 ticks
//				Delta 46 deg	= Delta 200 us = Delta 500 ticks (offset value = -67)
//				Multiplicator	= 1391
//				Shift			= 7
//				Base_tick	= 1250 (500us * 2000 / 800us)
//				Inversion		= 1
//
//				Neck Yaw #########################
//				50   	deg		= 1100us + 578us = 1678us = 63 840 ticks
//				130 deg		= 1100us + 222us = 1322us = 64 730 ticks
//				Delta 80 deg	= Delta 356 us = Delta 890 ticks (offset value = -50)
//				Multiplicator	= 89
//				Shift			= 3
//				Base_tick	= 1445 (578us * 2000 / 800us)
//				Inversion		= 1
//
//				Mouth Right Side (Up & Down) #########################
//				20 deg		= 1100us + 178us = 1278us = 64 840 ticks
//				70 deg		= 1100us + 622us = 1722us = 63 730 ticks
//				Delta 50 deg	= Delta 444 us = Delta 1110 ticks (offset value = -20)
//				Multiplicator	= 5683
//				Shift			= 8
//				Base_tick	= 445 (178us * 2000 / 800us)
//				Inversion		= 0
//
//				Mouth Left Side (Up & Down) #########################
//				20 deg		= 1100us + 622us = 1722us = 63 730 ticks
//				70 deg		= 1100us + 178us = 1278us = 64 840 ticks
//				Delta 50 deg	= Delta 444 us = Delta 1110 ticks (offset value = -20)
//				Multiplicator	= 5683
//				Shift			= 8
//				Base_tick	= 1555 (622us * 2000 / 800us)
//				Inversion		= 1
//
//				Eye Right Yaw  #################################
//				26 deg		= 1100us + 250us = 1350us = 64 660 ticks
//				64 deg		= 1100us + 590us = 1690us = 63 810 ticks
//				Delta 38 deg	= Delta 340 us = Delta 850 ticks (offset value = -26)
//				Multiplicator	= 2863
//				Shift			= 7
//				Base_tick	= 625 (250us * 2000 / 800us)
//				Inversion		= 0
//
//				Eye Left Yaw  #################################
//				26 deg		= 1100us + 210us = 1310us = 64 760 ticks
//				64 deg		= 1100us + 550us = 1650us = 63 910 ticks
//				Delta 38 deg	= Delta 340 us = Delta 850 ticks (offset value = -26)
//				Multiplicator	= 2863
//				Shift			= 7
//				Base_tick	= 525 (210us * 2000 / 800us)
//				Inversion		= 0
//
//				Eyes Pitch  #################################
//				25 deg		= 1100us + 222us = 1310us = 64 730 ticks
//				65 deg		= 1100us + 578us = 1650us = 63 840 ticks
//				Delta 40 deg	= Delta 356 us = Delta 890 ticks (offset value = -25)
//				Multiplicator	= 89
//				Shift			= 2
//				Base_tick	= 555 (222us * 2000 / 800us)
//				Inversion		= 0
//
//				Eyebrows Left #################################
//				10 deg		= 1100us + 89 us = 1189 us = 65 063 ticks
//				60 deg		= 1100us + 533 us = 1633us = 63 952 ticks
//				Delta 50 deg	= Delta 444 us = Delta 1111 ticks (offset value = -10)
//				Multiplicator	= 711
//				Shift			= 5
//				Base_tick	= 223 (89 us * 2000 / 800us)
//				Inversion		= 0
//
//				Eyebrows Right #################################
//				10 deg		= 1100us + 711 us = 1811 us = 63 507 ticks
//				60 deg		= 1100us + 267 us = 1367 us = 64 618 ticks
//				Delta 50 deg	= Delta 444 us = Delta 1111 ticks (offset value = -10)
//				Multiplicator	= 711
//				Shift			= 5
//				Base_tick	= 1778 (711 us * 2000 / 800us)
//				Inversion		= 1
//
//////////////////////////////////////////////////////////////////////
unsigned int norm_servo(unsigned char serial_value, unsigned short long multiplicator, unsigned char shift, unsigned int base_tick, unsigned char inversion)
{
	unsigned short long temp_value = 0;
	unsigned int output_result = 0;
	
	temp_value = serial_value;

	temp_value = temp_value * multiplicator;

	output_result = (unsigned int)(temp_value >> shift);
	
	if(inversion)
	{
		output_result = base_tick - output_result;
	}
	else
	{
		output_result += base_tick;
	}
	
	return(output_result);
}


//////////////////////////////////////////////////////////////////////
//   norm_servor_270 -> depricated
//////////////////////////////////////////////////////////////////////
//
//   Description: This function will normalize a value coming from serial
//                line.  Conversion 270 deg = 1900 ms & 0 deg = 1100 ms
//
//   Input: Unsigned char Serial_Value (to be converted)
//   Output: Unsigned char (converted value)
//   Input/Output:
//   Returned value: NONE
//   Comment : Since the serial protocol support onlyl unsigned char, 
//			   function is naturally limited to 0-255 deg.
//////////////////////////////////////////////////////////////////////
unsigned int norm_servo_270(unsigned char serial_value)
{
	unsigned short long temp_value = 0;
	unsigned int output_result = 0;

	temp_value = serial_value;

	temp_value = (temp_value * Basic_Multiplication_270) * Norm_Multiplication_270;

	output_result = (unsigned int)(temp_value >> Norm_Shifting_270);
	
	return(output_result);
}

//////////////////////////////////////////////////////////////////////
//   norm_servor_180 -> depricated
//////////////////////////////////////////////////////////////////////
//
//   Description: This function will normalize a value coming from serial
//                line.  Conversion 180 deg = 1900 ms & 0 deg = 1100 ms
//
//   Input: Unsigned char Serial_Value (to be converted)
//   Output: Unsigned char (converted value)
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
unsigned int norm_servo_180(unsigned char serial_value)
{
	unsigned short long temp_value = 0;
	unsigned int output_result = 0;

	temp_value = serial_value;

	temp_value = (temp_value * Basic_Multiplication_180) * Norm_Multiplication_180;

	output_result = (unsigned int)(temp_value >> Norm_Shifting_180);
	
	return(output_result);
}

//////////////////////////////////////////////////////////////////////
//   norm_servor_90 -> depricated
//////////////////////////////////////////////////////////////////////
//
//   Description: This function will normalize a value coming from serial
//                line.  Conversion 90 deg = 1900 ms & 0 deg = 1100 ms
//
//   Input: Unsigned char Serial_Value (to be converted)
//   Output: Unsigned char (converted value)
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
unsigned int norm_servo_90(unsigned char serial_value)
{
	unsigned short long temp_value = 0;
	unsigned int output_result = 0;

	temp_value = serial_value;

	temp_value = (temp_value * Basic_Multiplication_90) * Norm_Multiplication_90;

	output_result = (unsigned int)(temp_value >> Norm_Shifting_90);
	
	return(output_result);
}

//////////////////////////////////////////////////////////////////////
//   GET SERVO ID
//////////////////////////////////////////////////////////////////////
//
//   Description: Make the equivalence from I/O physical pins and virtual pint ID
//                
//
//   Input: unsigned char servo_alias (from physical pin ID : [1 to 20])
//   Output: unsigned char (virtual pin ID)
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
unsigned char get_servo_id(unsigned char servo_alias)
{
	unsigned char servo_id = 0xFF; //invalid servo_id

	//This section should be modified to integrate CONST from main.h
	//PL ; 2007-11-29

	switch(servo_alias){
		case 1:
			servo_id = 17;
			break;
		case 2:
			servo_id = 18;
			break;
		case 3:
			servo_id = 19;
			break;
		case 4:
			servo_id = 0;
			break;
		case 5:
			servo_id = 1;
			break;
		case 6:
			servo_id = 2;
			break;
		case 7:
			servo_id = 14;
			break;
		case 8:
			servo_id = 15;
			break;
		case 9:
			servo_id = 16;
			break;
		case 10:
			servo_id = 3;
			break;
		case 11:
			servo_id = 13;
			break;
		case 12:
			servo_id = 12;
			break;
		case 13:
			servo_id = 11;
			break;
		case 14:
			servo_id = 10;
			break;
		case 15:
			servo_id = 9;
			break;
		case 16:
			servo_id = 8;
			break;
		case 17:
			servo_id = 7;
			break;
		case 18:
			servo_id = 6;
			break;
		case 19:
			servo_id = 5;
			break;
		case 20:
			servo_id = 4;
			break;
	}
	return(servo_id);

}

//////////////////////////////////////////////////////////////////////
//   LIMIT POSITION
//////////////////////////////////////////////////////////////////////
//
//   Description: This function make sure that no unsafe value will be
//				  sent to the servo controller.
//                
//
//   Input: unsigned char servo_alias (from physical pin), unsigned value (value to be checked)
//   Output: unsigned int (limited value, or value)
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
unsigned int limit_position(unsigned char servo_alias, unsigned int value)
{
	unsigned int retval = 0;

	retval = value;

	switch(servo_alias){
		case ELBOW_RIGHT:
			// Elbow right 
			if(value < 0){
				retval = 0;
			}
			else if (value > 90){
				retval = 90;
			}
			
			break;
		case SHOULDER_YAW_RIGHT:
			// Right shoulder yaw

			if(value <0){
				retval = 0;
			}
			else if (value > 90){
				retval = 90;
			}
			break;
		case SHOULDER_PITCH_RIGHT:
			// Right shoulder pitch
			if(value < 0){
				retval = 0;
			}
			else if (value > 180){
				retval = 180;
			}
			break;
		case SHOULDER_PITCH_LEFT:
			// Left shoulder pitch
			if(value < 0){
				retval = 0;
			}
			else if (value > 180){
				retval = 180;
			}
			break;
		case 5:
			break;
		case ELBOW_LEFT:
			// Elbow left
			if(value < 0){
				retval = 0;
			}
			else if (value > 90){
				retval = 90;
			}
			
			break;
		case SHOULDER_YAW_LEFT:
			//Left shoulder yaw
			if(value < 0){
				retval = 0;
			}
			else if (value > 90){
				retval = 90;
			}
			break;
		case NECK_YAW:
			// Neck Yaw
			if(value < 50){
				retval = 50;
			}
			else if (value > 130){
				retval = 130;
			}
			break;
		case NECK_PITCH:
			// Neck Pitch
			if(value < 67){
				retval = 67;
			}
			else if (value > 113){
				retval = 113;
			}
			break;
		case 10:
			break;
		case MOUTH_UPPER_RIGHT:
			// Mouth upper left
			if(value < 20){
				retval = 20;
			}
			else if (value > 70){
				retval = 70;
			}
			break;
		case MOUTH_LOWER_LEFT:
			// Mouth bottom right
			if(value < 20){
				retval = 20;
			}
			else if (value > 70){
				retval = 70;
			}
			break;
		case MOUTH_UPPER_LEFT:
			// Mouth upper right
			if(value < 20){
				retval = 20;
			}
			else if (value > 70){
				retval = 70;
			}
			break;
		case MOUTH_LOWER_RIGHT:
			// Mouth bottom left
			if(value < 20){
				retval = 20;
			}
			else if (value > 70){
				retval = 70;
			}
			break;
		case EYES_PITCH:
			// Eyes Pitch
			if(value < 25){
				retval = 25;
			}
			else if (value > 65){
				retval = 65;
			}
			break;
		case EYE_YAW_LEFT:
			// Right Eye yaw
			if(value < 26){
				retval = 26;
			}
			else if (value > 64){
				retval = 64;
			}
			break;
		case EYE_YAW_RIGHT:
			// Left Eye yaw
			if(value < 26){
				retval = 26;
			}
			else if (value > 64){
				retval = 64;
			}
			break;
		case EYEBROW_RIGHT:
			// Left EyeBrow
			if(value < 10){
				retval = 10;
			}
			else if (value > 60){
				retval = 60;
			}
			break;
		case 19:
			break;
		case EYEBROW_LEFT:
			// Right EyeBrow
			if(value < 10){
				retval = 10;
			}
			else if (value > 60){
				retval = 60;
			}
			break;
	}

	return(retval);

}

//////////////////////////////////////////////////////////////////////
//   EnableHighInterrupts
//////////////////////////////////////////////////////////////////////
//
//   Description: Initialize the interrupt high level
//               
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void EnableHighInterrupts(void) {
	INTCONbits.GIEH = 1;
	INTCONbits.GIEL = 1;
}

//--------------------------------------- STANDARD FUNCTION ------------------------------------------------
//////////////////////////////////////////////////////////////////////
//   SEND WHISKER STATE
//////////////////////////////////////////////////////////////////////
//
//   Description: This function send the state of whisker on serial line
//				  
//                
//
//   Input: 
//   Output: 
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void send_whisker_state(void)
{
	char whisker_msg[4] = {0xff};

	whisker_msg[0] = 'g';
	whisker_msg[1] = 'w';
	whisker_msg[2] = 's';
	whisker_msg[3] = ctrl_panel.whisker_state;

	send_s_serial(whisker_msg,4);
// ----------------------------------------------------------
	//Modification 5 Dec 2006
	//ctrl_panel.whisker_state = 0;
// ----------------------------------------------------------
}

//////////////////////////////////////////////////////////////////////
//   SEND BATTERY STATE
//////////////////////////////////////////////////////////////////////
//
//   Description: This function send the state of battery on serial line
//				  
//                
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void send_battery_state(void)
{
	char batt_msg[5] = {0xff};

	batt_msg[0] = 'g';
	batt_msg[1] = 'b';
	batt_msg[2] = 'v';
	batt_msg[3] = ctrl_panel.battery_1_volt;
	batt_msg[4] = ctrl_panel.battery_2_volt;

	send_s_serial(batt_msg,5);
}

//////////////////////////////////////////////////////////////////////
//   SEND ANALOGUE STATE
//////////////////////////////////////////////////////////////////////
//
//   Description: This function send the state of analogue inputs on serial line
//				  
//                
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void send_analogue_state(void)
{
	char analogue_msg[6] = {0xff};
	char analogue_low = 0;
	char analogue_high = 0;

	analogue_msg[0] = 'g';
	analogue_msg[1] = 'a';
	analogue_msg[2] = 'v';
	analogue_msg[3] = 1;
	analogue_low = (ctrl_panel.analogue_1_value & 0x00FF);
	analogue_high = (ctrl_panel.analogue_1_value >> 8)  & 0x00FF;
	analogue_msg[4] = analogue_high;
	analogue_msg[5] = analogue_low;
	send_s_serial(analogue_msg,6);

	analogue_msg[3] = 2;
	analogue_low = 0;
	analogue_high = 0;
	analogue_low = (ctrl_panel.analogue_2_value  & 0x00FF);
	analogue_high = (ctrl_panel.analogue_2_value >> 8)  & 0x00FF;
	analogue_msg[4] = analogue_high;
	analogue_msg[5] = analogue_low;
	send_s_serial(analogue_msg,6);

	analogue_msg[3] = 3;
	analogue_low = 0;
	analogue_high = 0;
	analogue_low = (ctrl_panel.analogue_3_value  & 0x00FF);
	analogue_high = (ctrl_panel.analogue_3_value >> 8)  & 0x00FF;
	analogue_msg[4] = analogue_high;
	analogue_msg[5] = analogue_low;
	send_s_serial(analogue_msg,6);

	analogue_msg[3] = 4;
	analogue_low = 0;
	analogue_high = 0;
	analogue_low = (ctrl_panel.analogue_4_value  & 0x00FF);
	analogue_high = (ctrl_panel.analogue_4_value >> 8)  & 0x00FF;
	analogue_msg[4] = analogue_high;
	analogue_msg[5] = analogue_low;
	send_s_serial(analogue_msg,6);
}

//////////////////////////////////////////////////////////////////////
//   BATTERY_DETECTION
//////////////////////////////////////////////////////////////////////
//
//   Description: Acquire battery analogue input and convert it in volt*10
//				  i.e. : 78 units = 7.8 volts
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//
//   RED0104
//   		Ref : 4.55V => 1024
//   		Input Analogue : Batt / 2 (ex : if batt == 8 volts, then input analog is 4 volts)
//////////////////////////////////////////////////////////////////////
void battery_detection(void)
{
	//108 us
	ctrl_panel.battery_1_volt =  get_batt1();
	ctrl_panel.battery_2_volt =  get_batt2();
	
	//2.1 us
	ctrl_panel.battery_1_volt = (((ctrl_panel.battery_1_volt * 45) >> 9) & 0x00FF);
	ctrl_panel.battery_2_volt = (((ctrl_panel.battery_2_volt * 45) >> 9) & 0x00FF);
}

//////////////////////////////////////////////////////////////////////
//   ANALOGUE DETECTION
//////////////////////////////////////////////////////////////////////
//
//   Description: Acquire 4 analogue inputs AN1, AN2, AN3, AN4
//				 
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void analogue_detection(void)
{
	ctrl_panel.analogue_1_value = get_raw_input_1();
	ctrl_panel.analogue_2_value = get_raw_input_2();
	ctrl_panel.analogue_3_value = get_raw_input_3();
	ctrl_panel.analogue_4_value = get_raw_input_4();

}
//--------------------------------------- INITIALIZATION FUNCTION ------------------------------------------

//////////////////////////////////////////////////////////////////////
//   INIT CAN MSG
//////////////////////////////////////////////////////////////////////
//
//   Description: Initialize CAN msg track
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void init_can_msg(void)
{
	message_to_send.msg_priority = CAN_PRIORITY_HIGHEST;
	message_to_send.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY;			
	message_to_send.msg_read_write = 0;
	message_to_send.msg_eeprom_ram = 0;
	message_to_send.msg_cmd = CAN_SERVO_SET_POINT;
	message_to_send.msg_remote = 0;
	message_to_send.msg_data_length = 4;
	message_to_send.msg_dest = 0x27;

	message_debug.msg_priority = CAN_PRIORITY_HIGHEST;
	message_debug.msg_type = CAN_TYPE_ACTUATOR_HIGH_PRIORITY;
	message_debug.msg_read_write = 0;
	message_debug.msg_eeprom_ram = 0;
	message_debug.msg_cmd = CAN_SERVO_GET_POINT;
	message_debug.msg_remote = 0;
	message_debug.msg_data_length = 1;
}

//////////////////////////////////////////////////////////////////////
//   INIT CTRL PARAMS
//////////////////////////////////////////////////////////////////////
//
//   Description: Initialize the control structure
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void init_ctrl_params(void)
{
	*TRACER = 0;
	*COUNTER = 0;
	*RCREG_VALUE = 0;
	ctrl_panel.whisker_state = 0;
	ctrl_panel.whisker_requested = 0;
	ctrl_panel.battery_1_volt = 0;
	ctrl_panel.battery_2_volt = 0;
	ctrl_panel.battery_requested = 0;
	ctrl_panel.analogue_requested = 0;
	ctrl_panel.analogue_1_value = 0;
	ctrl_panel.analogue_2_value = 0;
	ctrl_panel.analogue_3_value = 0;
	ctrl_panel.analogue_4_value = 0;

	ctrl_panel.old_state_RB4 = 0xFF;
	ctrl_panel.old_state_RB5 = 0xFF;
	ctrl_panel.old_state_RB6 = 0xFF;
	ctrl_panel.old_state_RB7 = 0xFF;

}
//////////////////////////////////////////////////////////////////////
//   INIT RCV STRING
//////////////////////////////////////////////////////////////////////
//
//   Description: Initialize the receiving string structure
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void init_rcv_string(void)
{
	unsigned char i = 0;

	ctrl_string.str_id = 0;
	ctrl_string.str_index = 0;
	ctrl_string.str_header = 1;

	ctrl_string.str_params_index = 0;
	for(i=0;i<8;i++){
		ctrl_string.str_params_buffer[i] = 0;
	}
}

//////////////////////////////////////////////////////////////////////
//   INIT FACE
//////////////////////////////////////////////////////////////////////
//
//   Description: Initialize the receiving string structure
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void init_face(void)
{
	unsigned int CAN_degree = 0;
	unsigned int value_limited = 0;

	Delay10KTCYx(100); //wait 100 ms before initializing => give a break to power
//------------------ EYES ---------------------------------------------------
	//Servo 17 - 16 - 15 (0-90)
	//Doing RIGHT EYE YAW servo 17
	message_to_send.msg_data[0] = get_servo_id(17); //servo real id
	value_limited = limit_position(17,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-26, 2863,7,625,0);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));

	//Doing LEFT EYE YAW servo 16
	message_to_send.msg_data[0] = get_servo_id(16); //servo real id
	value_limited = limit_position(16,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-26, 2863,7,525,0);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));

	//Doing EYE PITCH servo 15
	message_to_send.msg_data[0] = get_servo_id(15); //servo real id
	value_limited = limit_position(15,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-25, 89,2,555,0);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));

//------------------ MOUTH ---------------------------------------------------
	Delay10KTCYx(100); //wait 100 ms before initializing the mouth => give a break to power
	//Servo 11 - 14 - 13 - 12 (0-90)
	//Doing MOUTH UPPER RIGHTservo 11
	message_to_send.msg_data[0] = get_servo_id(11); //servo real id
	value_limited = limit_position(11,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,445,0);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));

	//Doing MOUTH LOWER RIGHT servo 14
	message_to_send.msg_data[0] = get_servo_id(14); //servo real id
	value_limited = limit_position(14,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,445,0);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));

	//Doing MOUTH UPPER LEFT servo 13
	message_to_send.msg_data[0] = get_servo_id(13); //servo real id
	value_limited = limit_position(13,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,1555,1);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));

	//Doing MOUTH LOWER LEFT servo 12
	message_to_send.msg_data[0] = get_servo_id(12); //servo real id
	value_limited = limit_position(12,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-20, 5683,8,1555,1);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));
//------------------ EYEBROWS ---------------------------------------------------
	Delay10KTCYx(100); //wait 100 ms before initializing => give a break to power
	//Move Eyebrows
	//Servo 18 - 20 (0-90)
	//Doing EYEBROW RIGHT servo 18
	message_to_send.msg_data[0] = get_servo_id(18); //servo real id
	value_limited = limit_position(18,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-10, 711,5,1778,1);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));

	//Doing EYEBROW LEFT servo 20
	message_to_send.msg_data[0] = get_servo_id(20); //servo real id
	value_limited = limit_position(20,45);

	//CAN_degree = BASED_CAN_DEGREE - norm_servo_90(value_limited);
	CAN_degree = BASED_CAN_DEGREE - norm_servo(value_limited-10, 711,5,223,0);
	message_to_send.msg_data[1] = CAN_degree & 0x00FF;
	message_to_send.msg_data[2] = (CAN_degree >> 8) & 0x00FF;
	message_to_send.msg_data[3] = 0;
	while(can_send_laborius_packet(&message_to_send));
	Delay10KTCYx(100); //wait 100 ms before initializing
}

//////////////////////////////////////////////////////////////////////
//   MAIN
//////////////////////////////////////////////////////////////////////
//
//   Description: Initialize and run trough main loop so picoModule
//                keeps reading and updating whisker and battery state
//
//   Input: NONE
//   Output: NONE
//   Input/Output:
//   Returned value: NONE
//////////////////////////////////////////////////////////////////////
void main(void) {
	
   unsigned char i=0;
   LABORIUS_MASK mask_in[2];
   LABORIUS_FILTER filter_in[6];

   TRISAbits.TRISA4 = 0;
   PORTAbits.RA4 = 0;


   //READ CAN ADDRESS FROM  EEPROM
   	Can_Addr = can_read_eeprom(0xFE);
	Delay100TCYx(10);

   // init mask
   for(i=0;i<2;i++){
      mask_in[i].mask_priority = 0;
      mask_in[i].mask_type = 0;
      mask_in[i].mask_cmd = 0;
      mask_in[i].mask_dest = 0x00; //0xFF
   }

   // init filter
   for(i=0;i<6;i++){
      filter_in[i].filter_priority = 0;
      filter_in[i].filter_type = 0;
      filter_in[i].filter_cmd = 0;
      filter_in[i].filter_dest = Can_Addr;
   }

   // init can
   can_init(filter_in,mask_in);

   // MEMORY SECTION ----------------------------------------------------
	//Have a look to the eeprom to see if good params are there
	read_data_flow_table(0x00,READ_WRITE_6_BYTE,CAN_REQUEST_EEPROM,data_flow_buffer);
	if((data_flow_buffer[0] != 0x01) || (data_flow_buffer[1] != 0x13) || (data_flow_buffer[4] != 0x04) || (data_flow_buffer[5] != CAN_NORMAL_MODE_ID)){
		//Configuration of memory
	   data_flow_buffer[0] = 0x01;         	    	//Table version
	   data_flow_buffer[1] = 0x16;                  //Project ID
	   data_flow_buffer[2] = 0x00;                  //General ID
	   data_flow_buffer[3] = 0x00;                  //Module  ID
	   data_flow_buffer[4] = 0x01;                  //Code version
	   data_flow_buffer[5] = CAN_NORMAL_MODE_ID;    //Actual State
	
	   write_data_flow_table(0x00,READ_WRITE_6_BYTE,CAN_REQUEST_EEPROM,data_flow_buffer);
	}

// ######################## LOADING MEMORY CONFIGURATION #########################

// ######################### SOFTWARE INITIALISATION ######################################
	init_can_msg();
	init_rcv_string();
	init_ctrl_params();
// ######################### HARDWARE INITIALISATION ######################################
	init_analog();
	init_serial();
// ######################## INTERRUPTIONS  INITIALISATION ################################

	TRISDbits.TRISD7 = 0;
	PORTDbits.RD7 = 0;

	//Setup Timers
	OpenTimer0(TIMER_INT_ON & T0_16BIT & T0_SOURCE_INT & T0_PS_1_16);
//----------------------------------------------------------------------------------------
	//Modification 5 December 2006
	//Setup interrupt on PORTB
	//OpenPORTB(PORTB_CHANGE_INT_ON & PORTB_PULLUPS_OFF);
//----------------------------------------------------------------------------------------

	//Configure priority
	INTCON2bits.TMR0IP = 0; //Low
	INTCON2bits.RBIP = 0;	//Low
	IPR1bits.RCIP = 1;		//High

	//ENABLE INTERRUPT PRIORITY
	RCONbits.IPEN = 1;
	
	//Enable interrupts
	EnableHighInterrupts();

//SPECIAL INITIALIZATION-----------------------------------------------------------------
//will initialize correctly the servo controller (facial)
	init_face();


// ########################### MAIN LOOP ###################################################
	while(1) {
		can_transceiver(Can_Addr);

		if(ctrl_panel.whisker_requested == 1){
			ctrl_panel.whisker_requested = 0;
			//Send serial cmd to the user
			send_whisker_state();
		}

		if(ctrl_panel.battery_requested == 1){
			ctrl_panel.battery_requested = 0;
			//get batteries voltage
			battery_detection();
			send_battery_state();
		}

		if(ctrl_panel.analogue_requested ==1){
			ctrl_panel.analogue_requested = 0;
			analogue_detection();
			send_analogue_state();
		}
	}//End While
}
